let $ = jQuery;

import {
	restCall,
	isEmpty,
	isDefined
} from './utilities.js';

class ManageProgress {
	constructor(){
		// Check if there is a "manage progress" container
		if ( this.isManageProgress() ){
			// Check if the user can manage this progress
			if ( this.canManageProgress() ){
				// Set settings
				this.setSettings();

				// Call to the functions that will add event listeners to the checkboxes
				this.listenProgressChangesOnCourses();
			}
		}
	}

	setSettings(){
		this.settings = {}
	}

	listenProgressChangesOnCourses(){
		// Get all courses
		const $courses = $( '.ultp-dashboard-course' );

		// Create Courses
		$.each( $courses, ( index, $element ) => {
			// Create instance of Course
			new Course( $( $element ), this.settings );
		});
	}

	isManageProgress(){
		return $( '.ultp-dashboard--manage-progress' ).length > 0 && typeof ULGM_ManageProgress !== 'undefined';
	}

	canManageProgress(){
		return !! parseInt( ULGM_ManageProgress.canManageProgress );
	}
}

class Course {
	constructor( $course ){
		// Get elements
		this.getElements( $course );

		// Get course data
		this.getCourseData();

		// Create children
		this.createChildren();

		// Listen clicks on the checkboxes
		this.listenCheckboxChanges();

		// Set the default value for the confirmationVisible property
		this.confirmationVisible = false;
	}

	getElements( $course ){
		this.$elements = {
			course:         $course,
			actions:        $course.find( '.ultp-dashboard-course__progress-actions' ),
			checkbox:       $course.find( '.ultp-dashboard-course__progress-action-checkbox' ),
			cancel:         $course.find( '.ultp-dashboard-course__progress-action-cancel' ),
			progress: {
				number:     $course.find( '.ultp-dashboard-course__progress-percentage span' ),
				bar:        $course.find( '.ultp-dashboard-course__progress-bar' )
			},
			lessons:        $course.find( '.ultp-dashboard-lesson' ),
			quizzes:        $course.find( '.ultp-dashboard-quiz' ),
			topics:         $course.find( '.ultp-dashboard-topic' ),
			rightContainer: $course.find( '.ultp-dashboard-course__right' )
		}
	}

	getCourseData(){
		this.courseData = {
			courseId:       this.$elements.course.data( 'course-id' ),
			status:         this.$elements.course.data( 'status' ),
			hasLessons:     !! parseInt( this.$elements.course.data( 'has-lessons' ) ),
			hasQuizzes:     !! parseInt( this.$elements.course.data( 'has-quizzes' ) ),
			hasCertificate: !! parseInt( this.$elements.course.data( 'has-certificate' ) ),
		}
	}

	createChildren(){
		// Create empty arrays
		this.courseData.children = {
			lessons: [],
			quizzes: [],
			topics:  []
		}

		// Create lessons
		$.each( this.$elements.lessons, ( index, lesson ) => {
			// Get jQuery element
			const $lesson = $( lesson );

			// Create lesson instances
			this.courseData.children.lessons.push( new Lesson( this, $lesson ) );
		});

		// Create quizzes
		$.each( this.$elements.quizzes, ( index, quiz ) => {
			// Get jQuery element
			const $quiz = $( quiz );

			// Create lesson instances
			this.courseData.children.quizzes.push( new Quiz( this, $quiz ) );
		});

		// Create topics
		$.each( this.$elements.topics, ( index, topic ) => {
			// Get jQuery element
			const $topic = $( topic );

			// Create lesson instances
			this.courseData.children.topics.push( new Topic( this, $topic ) );
		});

		// Create object with all the children
		this.courseData.children.all = [].concat( this.courseData.children.lessons, this.courseData.children.topics, this.courseData.children.quizzes )
	}

	setProgress( percentage ){
		// Set new progress
		this.$elements.progress.number.text( percentage );
		this.$elements.progress.bar.css({ width: percentage });

		// Set correct status class. First, remove all of them
		this.$elements.course.removeClass( 'ultp-dashboard-course--not-started ultp-dashboard-course--in-progress ultp-dashboard-course--completed' );

		// Change status of the course
		if ( percentage == 100 ){
			// Update status
			this.courseData.status = 'completed';

			// Set status class
			this.$elements.course.addClass( 'ultp-dashboard-course--completed' );
		}
		else if ( percentage == 0 ){
			// Update status
			this.courseData.status = 'not-started';

			// Set status class
			this.$elements.course.addClass( 'ultp-dashboard-course--not-started' );
		}
		else {
			// Update status
			this.courseData.status = 'in-progress';

			// Set status class
			this.$elements.course.addClass( 'ultp-dashboard-course--in-progress' );
		}
	}

	setCertificate( certificateUrl ){
		// Find the container we have to remove
		this.$elements.rightContainer.find( '.ultp-dashboard-course__action' ).remove();

		// Check if we have to add the certificate button again
		if ( ! isEmpty( certificateUrl ) ){
			// Create the "Certificate" button
			const $certificateButton = $( `
				<div class="ultp-dashboard-course__action">
					<a href="${ certificateUrl }" target="_blank" class="ultp-dashboard-btn">
						${ ULGM_ManageProgress.i18n.certificate }
					</a>
				</div>
			` );

			// Add button to the container
			this.$elements.rightContainer.prepend( $certificateButton );
		}	
	}

	listenCheckboxChanges(){
		// Listen clicks on the checkbox
		this.$elements.checkbox.on( 'click', ( event ) => {
			// Check if we should do the change or if
			// we should show the confirmation buttons
			if ( this.confirmationVisible ){
				// Do request call
				let wantsToCompleteCourse = false;

				// Check the current status
				if ( [ 'not-started', 'in-progress' ].includes( this.courseData.status ) ){
					wantsToCompleteCourse = true;
				}

				// Add loading class to the actions container
				this.$elements.course.addClass( 'ultp-dashboard-course--loading' );
				this.$elements.actions.removeClass( 'ultp-dashboard-course__progress-actions--need-to-confirm' );

				// Do REST call
				restCall({
					url:      ULGM_ManageProgress.restUrl,
					endpoint: 'change_user_progress',
					nonce:    ULGM_ManageProgress.nonce,
					data:     {
						user_id:   ULGM_ManageProgress.userId,
						course_id: this.courseData.courseId,
						post_id:   this.courseData.courseId,
						type:      wantsToCompleteCourse ? 'complete_course' : 'uncomplete-course'
					},
					onDone: ( response ) => {
						// Remove loading and confirmation class to the actions container
						this.$elements.course.removeClass( 'ultp-dashboard-course--loading' );

						// Reset confirmation status
						this.confirmationVisible = false;
					},
					onSuccess: ( response ) => {
						// Check response
						if ( response.success ){
							// Set progress
							this.setProgress( response.progress.percentage );

							// Add certificate
							this.setCertificate( response.certificate_url );

							// Do the same with all the children
							this.completeCourseItems( response.progress.course_completed_children );
						}
					},
					onFail: ( response ) => {
						// Debug
						console.error( response );
					}
				});
			}
			else {
				// Show confirmation
				this.confirmChanges();
			}
		});
	}

	confirmChanges(){
		// Add class so the user knows that it has to confirm the change
		this.$elements.actions.addClass( 'ultp-dashboard-course__progress-actions--need-to-confirm' );

		// Change confirmation value
		this.confirmationVisible = true;

		// Bind cancel button
		this.$elements.cancel.on( 'click', () => {
			// Remove the class that shows both buttons without doing any other modification
			this.$elements.actions.removeClass( 'ultp-dashboard-course__progress-actions--need-to-confirm' );

			// Remove listener
			this.$elements.cancel.off( 'click' );

			// Change confirmation value
			this.confirmationVisible = false;
		});
	}

	completeCourseItems( completedChildren ){
		// Change the array to an object where the keys
		// are the IDs
		let completedChildrenObj = {}

		// Check if the completedChildren is empty
		completedChildren = Object.keys( completedChildren ).length == 0 ? [] : completedChildren;

		// Iterate all children
		completedChildren.forEach( ( element ) => {
			// Add children to the object
			completedChildrenObj[ element.ID ] = element;
		});

		// Iterate each children
		$.each( this.courseData.children.all, function( index, child ){
			// Check if the current children is included in the list of
			// completed children
			let shouldBeCompleted = isDefined( completedChildrenObj[ child.getTheId() ] );

			// Mark all the children as not completed
			child.markAsNotComplete( completedChildrenObj[ child.getTheId() ] );

			// And then check if we have to completed it
			if ( shouldBeCompleted ){
				// Complete
				child.markAsComplete( completedChildrenObj[ child.getTheId() ] );
			}
		});
	}
}

class Lesson {
	constructor( Course, $lesson ){
		// Save parent instance
		this.Course = Course;

		// Get elements
		this.getElements( $lesson );

		// Get lesson data
		this.getLessonData();

		// Listen to checkbox clicks
		this.listenCheckboxChanges();

		// Set the default value for the confirmationVisible property
		this.confirmationVisible = false;
	}

	getElements( $lesson ){
		this.$elements = {
			course:     $lesson.closest( '.ultp-dashboard-course' ),
			lesson:     $lesson,
			actions:    $lesson.find( '.ultp-dashboard-lesson__progress-actions' ),
			checkbox:   $lesson.find( '.ultp-dashboard-lesson__progress-action-checkbox' ),
			cancel:     $lesson.find( '.ultp-dashboard-lesson__progress-action-cancel' ),
		}
	}

	getLessonData(){
		this.lessonData = {
			lessonId:       this.$elements.lesson.data( 'lesson-id' ),
			courseId:       this.$elements.lesson.data( 'course-id' ),
			isCompleted:    this.$elements.lesson.data( 'is-completed' ),
			hasTopics:      !! parseInt( this.$elements.lesson.data( 'has-topics' ) ),
			hasQuizzes:     !! parseInt( this.$elements.lesson.data( 'has-quizzes' ) ),
		}

		// Check if it has children
		this.lessonData.hasChildren = this.lessonData.hasTopics || this.lessonData.hasQuizzes;
	}

	listenCheckboxChanges(){
		// Listen clicks on the checkbox
		this.$elements.checkbox.on( 'click', ( event ) => {
			// Check if we should do the change or if
			// we should show the confirmation buttons
			if ( ! this.lessonData.hasChildren || this.confirmationVisible ){
				// Do request call

				// Check if it wants to complete or not the lesson
				// If the lesson is not completed then the user wants to complete it, if
				// it's completed then it wants to mark it as incomplete
				let wantsToCompleteLesson = ! this.lessonData.isCompleted;

				// Add loading class to the actions container
				this.$elements.lesson.addClass( 'ultp-dashboard-lesson--loading' );
				this.$elements.actions.removeClass( 'ultp-dashboard-lesson__progress-actions--need-to-confirm' );

				// Do REST call
				restCall({
					url:      ULGM_ManageProgress.restUrl,
					endpoint: 'change_user_progress',
					nonce:    ULGM_ManageProgress.nonce,
					data:     {
						user_id:   ULGM_ManageProgress.userId,
						course_id: this.lessonData.courseId,
						post_id:   this.lessonData.lessonId,
						type:      wantsToCompleteLesson ? 'complete_post' : 'uncomplete_post'
					},
					onDone: ( response ) => {
						// Remove loading and confirmation class to the actions container
						this.$elements.lesson.removeClass( 'ultp-dashboard-lesson--loading' );

						// Update isCompleted property
						this.lessonData.isCompleted = wantsToCompleteLesson;

						// Reset confirmation status
						this.confirmationVisible = false;
					},
					onSuccess: ( response ) => {
						// Check response
						if ( response.success ){
							// Set progress
							this.Course.setProgress( response.progress.percentage );

							// Set progress
							this.Course.setCertificate( response.certificate_url );

							// Do the same with all the children
							this.Course.completeCourseItems( response.progress.course_completed_children );
						}
					},
					onFail: ( response ) => {
						// Debug
						console.error( response );
					}
				});
			}
			else {
				// Show confirmation
				this.confirmChanges();
			}
		});
	}

	confirmChanges(){
		// Add class so the user knows that it has to confirm the change
		this.$elements.actions.addClass( 'ultp-dashboard-lesson__progress-actions--need-to-confirm' );

		// Change confirmation value
		this.confirmationVisible = true;

		// Bind cancel button
		this.$elements.cancel.on( 'click', () => {
			// Remove the class that shows both buttons without doing any other modification
			this.$elements.actions.removeClass( 'ultp-dashboard-lesson__progress-actions--need-to-confirm' );

			// Remove listener
			this.$elements.cancel.off( 'click' );

			// Change confirmation value
			this.confirmationVisible = false;
		});
	}

	markAsComplete(){
		// Change the status class
		this.$elements.lesson.removeClass( 'ultp-dashboard-lesson--not-completed' );
		this.$elements.lesson.addClass( 'ultp-dashboard-lesson--completed' );

		// Change data status
		this.$elements.lesson.data( 'is-completed', true );
		this.lessonData.isCompleted = true;
	}

	markAsNotComplete(){
		// Change the status class
		this.$elements.lesson.removeClass( 'ultp-dashboard-lesson--completed' );
		this.$elements.lesson.addClass( 'ultp-dashboard-lesson--not-completed' );

		// Change data status
		this.$elements.lesson.data( 'is-completed', false );
		this.lessonData.isCompleted = false;
	}

	isComplete(){
		return this.lessonData.isCompleted;
	}

	getTheId(){
		return this.lessonData.lessonId;
	}
}

class Topic {
	constructor( Course, $topic ){
		// Save parent instance
		this.Course = Course;

		// Get elements
		this.getElements( $topic );

		// Get topic data
		this.getTopicData();

		// Listen to checkbox clicks
		this.listenCheckboxChanges();

		// Set the default value for the confirmationVisible property
		this.confirmationVisible = false;
	}

	getElements( $topic ){
		this.$elements = {
			course:     $topic.closest( '.ultp-dashboard-course' ),
			lesson:     $topic.closest( '.ultp-dashboard-lesson' ),
			topic:      $topic,
			actions:    $topic.find( '.ultp-dashboard-topic__progress-actions' ),
			checkbox:   $topic.find( '.ultp-dashboard-topic__progress-action-checkbox' ),
			cancel:     $topic.find( '.ultp-dashboard-topic__progress-action-cancel' ),
		}
	}

	getTopicData(){
		this.topicData = {
			topicId:        this.$elements.topic.data( 'topic-id' ),
			lessonId:       this.$elements.topic.data( 'lesson-id' ),
			courseId:       this.$elements.topic.data( 'course-id' ),
			isCompleted:    this.$elements.topic.data( 'is-completed' ),
			hasChildren:    false
		}
	}

	listenCheckboxChanges(){
		// Listen clicks on the checkbox
		this.$elements.checkbox.on( 'click', ( event ) => {
			// Check if we should do the change or if
			// we should show the confirmation buttons
			if ( ! this.topicData.hasChildren || this.confirmationVisible ){
				// Do request call

				// Check if it wants to complete or not the topic
				// If the topic is not completed then the user wants to complete it, if
				// it's completed then it wants to mark it as incomplete
				let wantsToCompleteTopic = ! this.topicData.isCompleted;

				// Add loading class to the actions container
				this.$elements.topic.addClass( 'ultp-dashboard-topic--loading' );
				this.$elements.actions.removeClass( 'ultp-dashboard-topic__progress-actions--need-to-confirm' );

				// Do REST call
				restCall({
					url:      ULGM_ManageProgress.restUrl,
					endpoint: 'change_user_progress',
					nonce:    ULGM_ManageProgress.nonce,
					data:     {
						user_id:   ULGM_ManageProgress.userId,
						course_id: this.topicData.courseId,
						post_id:   this.topicData.topicId,
						type:      wantsToCompleteTopic ? 'complete_post' : 'uncomplete_post'
					},
					onDone: ( response ) => {
						// Remove loading and confirmation class to the actions container
						this.$elements.topic.removeClass( 'ultp-dashboard-topic--loading' );

						// Update isCompleted property
						this.topicData.isCompleted = wantsToCompleteTopic;

						// Reset confirmation status
						this.confirmationVisible = false;
					},
					onSuccess: ( response ) => {
						// Check response
						if ( response.success ){
							// Set progress
							this.Course.setProgress( response.progress.percentage );

							// Set progress
							this.Course.setCertificate( response.certificate_url );

							// Do the same with all the children
							this.Course.completeCourseItems( response.progress.course_completed_children );
						}
					},
					onFail: ( response ) => {
						// Debug
						console.error( response );
					}
				});
			}
			else {
				// Show confirmation
				this.confirmChanges();
			}
		});
	}

	confirmChanges(){
		// Add class so the user knows that it has to confirm the change
		this.$elements.actions.addClass( 'ultp-dashboard-topic__progress-actions--need-to-confirm' );

		// Change confirmation value
		this.confirmationVisible = true;

		// Bind cancel button
		this.$elements.cancel.on( 'click', () => {
			// Remove the class that shows both buttons without doing any other modification
			this.$elements.actions.removeClass( 'ultp-dashboard-topic__progress-actions--need-to-confirm' );

			// Remove listener
			this.$elements.cancel.off( 'click' );

			// Change confirmation value
			this.confirmationVisible = false;
		});
	}

	markAsComplete(){
		// Change the status class
		this.$elements.topic.removeClass( 'ultp-dashboard-topic--not-completed' );
		this.$elements.topic.addClass( 'ultp-dashboard-topic--completed' );

		// Change data status
		this.$elements.topic.data( 'is-completed', true );
		this.topicData.isCompleted = true;
	}

	markAsNotComplete(){
		// Change the status class
		this.$elements.topic.removeClass( 'ultp-dashboard-topic--completed' );
		this.$elements.topic.addClass( 'ultp-dashboard-topic--not-completed' );

		// Change data status
		this.$elements.topic.data( 'is-completed', false );
		this.topicData.isCompleted = false;
	}

	isComplete(){
		return this.topicData.isCompleted;
	}

	getTheId(){
		return this.topicData.topicId;
	}
}

class Quiz {
	constructor( Course, $quiz ){
		// Save parent instance
		this.Course = Course;

		// Get elements
		this.getElements( $quiz );

		// Get quiz data
		this.getQuizData();

		// Listen to checkbox changes
		this.listenCheckboxChanges();
	}

	getElements( $quiz ){
		this.$elements = {
			course:         $quiz.closest( '.ultp-dashboard-course' ),
			quiz:           $quiz,
			actions:        $quiz.find( '.ultp-dashboard-quiz__progress-actions' ),
			checkbox:       $quiz.find( '.ultp-dashboard-quiz__progress-action-checkbox' ),
			cancel:         $quiz.find( '.ultp-dashboard-quiz__progress-action-cancel' ),
			rightContainer: $quiz.find( '.ultp-dashboard-quiz__right' )
		}
	}

	getQuizData(){
		this.quizData = {
			quizId:        this.$elements.quiz.data( 'quiz-id' ),
			lessonId:       this.$elements.quiz.data( 'lesson-id' ),
			courseId:       this.$elements.quiz.data( 'course-id' ),
			isCompleted:    this.$elements.quiz.data( 'is-completed' ),
		}
	}

	setCertificate( certificateUrl = null ){
		// Find the container we have to remove
		this.$elements.rightContainer.find( '.ultp-dashboard-quiz__action' ).remove();

		// Check if we have to add the certificate button again
		if ( ! isEmpty( certificateUrl ) ){
			// Create the "Certificate" button
			const $certificateButton = $( `
				<div class="ultp-dashboard-quiz__action">
					<a href="${ certificateUrl }" target="_blank" class="ultp-dashboard-btn">
						${ ULGM_ManageProgress.i18n.certificate }
					</a>
				</div>
			` );

			// Add button to the container
			this.$elements.rightContainer.append( $certificateButton );
		}	
	}

	listenCheckboxChanges(){
		// Create quiz data object
		let quizData = {
			user_id:   ULGM_ManageProgress.userId,
			course_id: this.quizData.courseId,
			post_id:   this.quizData.quizId
		}

		// Check if we can add the lesson id
		if ( ! isEmpty( this.quizData.lessonId ) ){
			// Add lesson id
			quizData.lesson_id = this.quizData.lessonId;
		}

		// Listen clicks on the checkbox
		this.$elements.checkbox.on( 'click', ( event ) => {
			// Do request call

			// Check if it wants to complete or not the quiz
			// If the quiz is not completed then the user wants to complete it, if
			// it's completed then it wants to mark it as incomplete
			let wantsToCompleteQuiz = ! this.quizData.isCompleted;

			// Add loading class to the actions container
			this.$elements.quiz.addClass( 'ultp-dashboard-quiz--loading' );
			this.$elements.actions.removeClass( 'ultp-dashboard-quiz__progress-actions--need-to-confirm' );

			// Check what action the user wants to do
			quizData.type = wantsToCompleteQuiz ? 'complete_quiz' : 'uncomplete_quiz';

			// Do REST call
			restCall({
				url:      ULGM_ManageProgress.restUrl,
				endpoint: 'change_user_progress',
				nonce:    ULGM_ManageProgress.nonce,
				data:     quizData,
				onDone: ( response ) => {
					// Remove loading and confirmation class to the actions container
					this.$elements.quiz.removeClass( 'ultp-dashboard-quiz--loading' );

					// Update isCompleted property
					this.quizData.isCompleted = wantsToCompleteQuiz;

					// Reset confirmation status
					this.confirmationVisible = false;
				},
				onSuccess: ( response ) => {
					// Check response
					if ( response.success ){
						// Set progress
						this.Course.setProgress( response.progress.percentage );

						// Set progress
						this.Course.setCertificate( response.certificate_url );

						// Set certificate
						this.setCertificate( response.certificate_url );

						// Do the same with all the children
						this.Course.completeCourseItems( response.progress.course_completed_children );
					}
				},
				onFail: ( response ) => {
					// Debug
					console.error( response );
				}
			});
		});
	}

	markAsComplete( childrenData ){
		// Remove certificate
		this.setCertificate( childrenData.certificate_url );

		// Change the status class
		this.$elements.quiz.removeClass( 'ultp-dashboard-quiz--not-completed' );
		this.$elements.quiz.addClass( 'ultp-dashboard-quiz--completed' );

		// Change data status
		this.$elements.quiz.data( 'is-completed', true );
		this.quizData.isCompleted = true;
	}

	markAsNotComplete( childrenData ){
		// Remove certificate
		this.setCertificate();

		// Change the status class
		this.$elements.quiz.removeClass( 'ultp-dashboard-quiz--completed' );
		this.$elements.quiz.addClass( 'ultp-dashboard-quiz--not-completed' );

		// Change data status
		this.$elements.quiz.data( 'is-completed', false );
		this.quizData.isCompleted = false;
	}

	isComplete(){
		return this.quizData.isCompleted;
	}

	getTheId(){
		return this.quizData.quizId;
	}
}

export default ManageProgress;