<?php
/**
 * The Admin Side LearnDash Slack
 *
 * @since		1.0.0
 *
 * @package LearnDash_Slack
 * @subpackage LearnDash_Slack/core/admin
 */

defined( 'ABSPATH' ) || die();

final class LearnDash_Slack_Admin {

	/**
	 * LearnDash_Slack_Admin constructor.
	 * 
	 * @since		1.0.0
	 */
	function __construct() {
		
		// Creates a (temporary) Submenu Item for our Admin Page
		add_action( 'admin_menu', array( $this, 'admin_menu' ) );
		
		// Register our Fields as WP Settings
		add_action( 'admin_init', array( $this, 'register_options' ) );
		
		// Creates the Webhook Repeater
		add_action( 'ld_slack_webhooks_field', array( $this, 'webhook_repeater_field' ) );
		
		// Creates the Repeater
		add_action( 'ld_slack_notifications_field', array( $this, 'notification_repeater_field' ) );
		
		// Include Hidden Field for Post ID within the Repeater
		add_action( 'ld_slack_slack_post_id', array( $this, 'post_id_field' ) );
		
		// Include Hidden Field for Term ID within the Repeater
		add_action( 'ld_slack_webhook_term_id', array( $this, 'term_id_field' ) );
		
		// Include Replacement Hints
		add_action( 'ld_slack_replacement_hints', array( $this, 'replacement_hints_field' ) );
		
		// Localize the admin.js
		add_filter( 'ld_slack_localize_admin_script', array( $this, 'localize_script' ) );
		
		// Enqueue our Styles/Scripts on our Settings Page
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
		
		// Adds Tooltips to our Settings
		add_action( 'ld_slack_after_setting_label', array( $this, 'add_tooltip' ), 10 );
		
		// Fix <title> tag for the Settings Page
		add_filter( 'admin_title', array( $this, 'admin_title' ), 10, 2 );
		
		// Add saving indicator to Timestamp Format
		add_action( 'ld_slack_after_setting_input_html', array( $this, 'add_saving_to_timestamp_format' ) );
		
	}
	
	/**
	 * Creates a (temporary) Submenu Item for our Admin Page
	 * 
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function admin_menu() {
		
		// We're hooking into options-general.php so that we have better access to WP's Settings API
		$page_hook = add_submenu_page(
			'options-general.php',
			_x( 'Slack', 'Slack Tab Label', 'learndash-slack' ),
			_x( 'Slack', 'Slack Tab Label', 'learndash-slack' ),
			'manage_options',
			'learndash_slack',
			array( $this, 'admin_page' )
		);
		
		global $submenu;
	
		// Ensure that we don't have errors from a non-existing Index for non-Admins
		if ( isset( $submenu['options-general.php'] ) ) {
			
			$settings_index = null;
			foreach ( $submenu['options-general.php'] as $key => $menu_item ) {

				// Index 2 is always the child page slug
				if ( $menu_item[2] == 'learndash_slack' ) {
					$settings_index = $key;
					break;
				}

			}

			// Unset from the Submenu
			unset( $submenu['options-general.php'][ $settings_index ] );

		}
		
	}
	
	/**
	 * Output our Admin Page (Finally!)
	 * 
	 * @access		public
	 * @since		1.0.0
	 * @return		HTML
	 */
	public function admin_page() {
			
		settings_errors(); ?>

		<div id="tab_container">

			<form method="post" action="options.php">
				
				<?php echo wp_nonce_field( 'learndash_slack_settings', 'learndash_slack_nonce' ); ?>

				<?php settings_fields( 'learndash_slack' ); ?>

				<?php do_settings_sections( 'learndash_slack' ); ?>

				<?php submit_button(); ?>

			</form>

		</div>

		<?php
		
	}
	
	/**
	 * Regsiter Options for each Field
	 * 
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function register_options() {
		
		if ( false === get_option( 'learndash_slack' ) ) {
			add_option( 'learndash_slack' );
		}
		
		add_settings_section(
			'learndash_slack',
			__return_null(),
			'__return_false',
			'learndash_slack'
		);
		
		// Holds all non-repeater values
		$global_values = get_option( 'learndash_slack' );
		
		// Initialize repeater
		$repeater_values = array();
		$fields = LEARNDASHSLACK()->get_settings_fields();
		$notification_id = apply_filters( 'ld_slack_notification_id', 'rbm' );
		
		$feeds = get_posts( array(
			'post_type'   => "ld-slack-{$notification_id}-feed",
			'numberposts' => -1,
			'order'	   => 'ASC',
		) );

		if ( ! empty( $feeds ) && ! is_wp_error( $feeds ) ) {

			foreach ( $feeds as $feed ) {

				$value = array(
					'admin_title' => get_the_title( $feed->ID ), // The first element in this Array is used for the Collapsable Title
					'slack_post_id' => $feed->ID,
				);
				
				// Conditionally Hide certain fields
				$trigger = get_post_meta( $feed->ID, 'ld_slack_rbm_feed_trigger', true );
				$trigger = ( $trigger ) ? $trigger : 0;

				foreach ( $fields as $field_id => $field ) {
					
					if ( $field_id == 'slack_post_id' || $field_id == 'admin_title' ) continue; // We don't need to do anything special with these
					
					$value[ $field_id ] = get_post_meta( $feed->ID, "ld_slack_rbm_feed_$field_id", true );
					
					if ( $field_id == 'replacement_hints' ) {
						
						$value[ $field_id ] = $trigger;
						
					}
					
					if ( $field['type'] == 'select' &&
						isset( $field['multiple'] ) && 
						$field['multiple'] === true ) {
						
						// Support for LD Slack v1.1.X
						$value[ $field_id ] = ( ! is_array( $value[ $field_id ] ) ) ? array( $value[ $field_id ] ) : $value[ $field_id ];
						
					}
					
				}

				$repeater_values[] = $value;

			}

		}
		
		$webhooks = get_terms( array(
			'taxonomy' => "ld-slack-{$notification_id}-webhook",
			'hide_empty' => false,
			'order' => 'DESC',
			'orderby' => 'date',
		) );
		
		$webhook_default_values = array();
		$webhook_repeater_values = array();
		
		foreach ( $webhooks as $webhook ) {
			
			$webhook_default_values[ $webhook->term_id ] = $webhook->name;
			
			$repeater_value = array(
				'webhook_term_id' => $webhook->term_id,
				'name' => $webhook->name,
				'url' => get_term_meta( $webhook->term_id, 'webhook_url', true ),
			);
			
			$webhook_repeater_values[] = $repeater_value;
			
		}
		
		$fields = array(
			array(
				'type' => 'hook',
				'id' => 'webhooks_field',
				'input_name' => 'ld_slack_rbm_webhooks',
				'settings_label' => _x( 'Slack Webhook URLs', 'Slack Webhook URLs Repeater Label', 'learndash-slack' ),
				'fields' => array(
					'webhook_term_id' => array(
						'type' => 'hook',
						'std' => '',
					),
					'name' => array(
						'type' => 'text',
						'label' => _x( 'Webhook Name', 'Slack Webhook Name Label', 'learndash-slack' ),
						'std' => '',
						'classes' => 'ld-slack-webhook-name',
						'pattern_message' => _x( 'Webhook Names must not match other names in use', 'Slack Webhook Name invalid message', 'learndash-slack' ),
					),
					'url' => array(
						'type' => 'text',
						'label' => _x( 'Webhook URL', 'Slack Webhook URL Label', 'learndash-slack' ),
						'std' => '',
						'classes' => 'ld-slack-webhook-url',
						'pattern' => 'https:\/\/hooks.slack.com\/services\/.*',
						'pattern_message' => _x( 'Must be a valid Slack Webhook URL', 'Slack Webhook URL invalid message', 'learndash-slack' ),
					),
				),
				'std' => $webhook_repeater_values,
				'desc' => sprintf(
					_x( 'Enter the Webhook URLs you would like to use here. The Default Webhook URL is the one that will be used as a fallback if one is not defined within a Notification. You can store as many Webhook URLs as you would like, which allows you to easily do things such as send Notifications to other Slack Teams. You should name them something easy for you to recognize as that is how they will be referred to within individual Notifications. If a Webhook URL that is in use by a Notification is deleted, it will fallback to the Default Webhook URL. You can set up Webhook URLs %shere%s.', 'Webhook URL Repeater Help Text', 'learndash-slack' ),
					'<a href="//my.slack.com/services/new/incoming-webhook/" target="_blank">',
					'</a>'
				),
				'add_item_text' => _x( 'Add Slack Webhoook URL', 'Add Slack Webhoook URL Button', 'learndash-slack' ),
				'edit_item_text' => _x( 'Edit', 'Edit Slack Webhoook URL Button', 'learndash-slack' ),
				'save_item_text' => _x( 'Save', 'Save Slack Webhoook URL Button', 'learndash-slack' ),
				'saving_item_text' => _x( 'Saving...', 'Saving Slack Webhoook URL Text', 'learndash-slack' ),
				'delete_item_text' => _x( 'Delete', 'Delete Slack Webhoook URL Button', 'learndash-slack' ),
			),
			array(
				'type' => 'text',
				'settings_label' => _x( 'Timestamp Format', 'Timestamp Format Label', 'learndash-slack' ),
				'input_name' => 'learndash_slack',
				'id' => 'timestamp_format',
				'std' => ( isset( $global_values['timestamp_format'] ) ) ? $global_values['timestamp_format'] : '',
				'desc' => _x( '<a href="//php.net/manual/en/function.date.php" target="_blank">Click Here</a> for Format Options. This applies to all <strong>%timestamp%</strong> Replacements', 'Timestamp Format Help Text', 'learndash-slack' ),
				'classes' => 'ld-slack-timestamp-format',
				'placeholder' => get_option( 'date_format', 'm/d/Y' ) . ' @ ' . get_option( 'time_format', 'g:i A' ),
			),
			array(
				'type' => 'hook',
				'id' => 'notifications_field',
				'input_name' => 'ld_slack_rbm_feeds',
				'std' => $repeater_values,
				'settings_label' => _x( 'Slack Notifications', 'Slack Notifications Repeater Label', 'learndash-slack' ),
				'sortable' => false,
				'collapsable' => true,
				'layout' => 'row',
				'add_item_text' => _x( 'Add Slack Notification', 'Add Slack Notification Button', 'learndash-slack' ),
				'edit_item_text' => _x( 'Edit Slack Notification', 'Edit Slack Notification Button', 'learndash-slack' ),
				'save_item_text' => _x( 'Save Slack Notification', 'Save Slack Notification Button', 'learndash-slack' ),
				'saving_item_text' => _x( 'Saving...', 'Saving Slack Notification Text', 'learndash-slack' ),
				'delete_item_text' => _x( 'Delete Slack Notification', 'Delete Slack Notification Button', 'learndash-slack' ),
				'default_title' => _x( 'New Slack Notification', 'New Slack Notification Header', 'learndash-slack' ),
				'fields' => $fields,
			),
		);
		
		foreach ( $fields as $field ) {
			
			$field = wp_parse_args( $field, array(
				'settings_label' => '',
			) );
			
			$callback = 'ld_slack_' . $field['type'] . '_callback';
			
			add_settings_field(
				$field['id'],
				$field['settings_label'],
				( is_callable( $callback ) ) ? 'ld_slack_' . $field['type'] . '_callback' : 'ld_slack_missing_callback',
				'learndash_slack',
				'learndash_slack',
				$field
			);
			
		}
		
		register_setting( 'learndash_slack', 'learndash_slack' );
		
	}
	
	/**
	 * Creates the Webhook Repeater
	 * 
	 * @param		array $args Field Args
	 *                           
	 * @access		public
	 * @since		1.2.0
	 * @return		string HTML
	 */
	public function webhook_repeater_field( $args ) {
		
		$args = wp_parse_args( $args, array(
			'id' => '',
			'std' => '',
			'classes' => array(),
			'fields' => array(),
			'add_item_text' => __( 'Add Row', 'learndash-slack' ),
			'edit_item_text' => __( 'Edit Row', 'learndash-slack' ),
			'save_item_text' => __( 'Save Row', 'learndash-slack' ),
			'saving_item_text' => __( 'Saving...', 'learndash-slack' ),
			'delete_item_text' => __( 'Delete Row', 'learndash-slack' ),
			'default_title' => __( 'New Row', 'learndash-slack' ),
			'input_name' => false,
		) );
		
		// Ensure Dummy Field is created
		$field_count = ( count( $args['std'] ) >= 1 ) ? count( $args['std'] ) : 1;
		
		$name = $args['input_name'] !== false ? $args['input_name'] : esc_attr( $args['name_attr_wrap'] ) . '[' . esc_attr( $args['id'] ) . ']';
		
		?>

			<div data-ld-slack-rbm-webhook-repeater class="ld-slack-rbm-webhook-repeater-container">
				
				<?php // We are masquerading as a Repeater Item, but we aren't really one. This is only to help inherit styling ?>
				<div class="ld-slack-rbm-webhook-repeater-default ld-slack-rbm-webhook-repeater-item">
					
					<div class="field-group">
						<?php // This is where the Term ID normally goes. This is just for spacing ?>
					</div>
				
					<div class="field-group">
						<?php ld_slack_text_callback( array(
							'type' => 'text',
							'label' => _x( 'Webhook Name', 'Default Webhook Name Label', 'learndash-slack' ) . '<br />',
							'id' => 'webhook_default_name_dummy',
							'std' => __( 'Default Webhook', 'learndash-slack' ),
							'disabled' => true,
						) ); ?>
					</div>
					
					<?php // Holds all non-repeater values
					$global_values = get_option( 'learndash_slack' ); ?>

					<div class="field-group">
						<?php ld_slack_text_callback( array(
							'type' => 'text',
							'label' => _x( 'Webhook URL', 'Default Webhook URL Label', 'learndash-slack' ) . '<br />',
							'input_name' => 'learndash_slack',
							'id' => 'webhook_default',
							'classes' => 'ld-slack-webhook-default',
							'std' => ( isset( $global_values['webhook_default'] ) ) ? $global_values['webhook_default'] : '',
							'pattern' => 'https:\/\/hooks.slack.com\/services\/.*',
							'pattern_message' => _x( 'Must be a valid Slack Webhook URL', 'Slack Webhook URL invalid message', 'learndash-slack' ),
						) ); ?>
					</div>
					
					<div class="field-group saving with-top-margin" style="display: none;">
						<span class="dashicons dashicons-clock"></span>
					</div>
					
				</div>
				
				<div data-repeater-list="<?php echo $name; ?>" class="ld-slack-rbm-webhook-repeater <?php echo ( isset( $args['classes'] ) ) ? ' ' . implode( ' ', $args['classes'] ) : ''; ?>">

					<?php for ( $index = 0; $index < $field_count; $index++ ) : $value = ( isset( $args['std'][$index] ) ) ? $args['std'][$index] : array(); ?>

						<div data-repeater-item<?php echo ( ! isset( $args['std'][$index] ) ) ? ' data-repeater-dummy style="display: none;"' : ''; ?> class="ld-slack-rbm-webhook-repeater-item">

							<?php foreach ( $args['fields'] as $field_id => $field ) : ?>

								<div class="field-group">

									<?php if ( is_callable( "ld_slack_{$field['type']}_callback" ) ) : 

										$field['id'] = $field_id;
										$field['std'] = ( isset( $value[ $field_id ] ) ) ? $value[ $field_id ] : $field['std'];

										if ( $field['type'] == 'checkbox' ) : 

											if ( isset( $field['std'] ) && (int) $field['std'] !== 0 ) {
												$field['field_class'][] = 'default-checked';
											}

										endif;

										call_user_func( "ld_slack_{$field['type']}_callback", $field ); 

									else :

										call_user_func( 'ld_slack_missing_callback', $field );

									endif; ?>
									
								</div>

							<?php endforeach; ?>
							
							<input data-repeater-delete type="button" class="button button-danger" value="<?php echo $args['delete_item_text']; ?>" />
							
							<div class="field-group saving with-top-margin" style="display: none;">
								<span class="dashicons dashicons-clock"></span>
							</div>

						</div>

					<?php endfor; ?>

				</div>
				
				<input data-repeater-create type="button" class="button" style="margin-top: 6px;" value="<?php echo $args['add_item_text']; ?>" />
				
			</div>

		<?php if ( ! empty( $args['desc'] ) ) : ?>
			<p class="description">
				<?php echo $args['desc']; ?>
			</p>
		<?php endif;
		
	}
	
	/**
	 * Creates the Notification Repeater
	 * 
	 * @param		array $args Field Args
	 *                           
	 * @access		public
	 * @since		1.1.0
	 * @return		string HTML
	 */
	public function notification_repeater_field( $args ) {
		
		$args = wp_parse_args( $args, array(
			'id' => '',
			'std' => '',
			'classes' => array(),
			'fields' => array(),
			'add_item_text' => __( 'Add Row', 'learndash-slack' ),
			'edit_item_text' => __( 'Edit Row', 'learndash-slack' ),
			'save_item_text' => __( 'Save Row', 'learndash-slack' ),
			'saving_item_text' => __( 'Saving...', 'learndash-slack' ),
			'delete_item_text' => __( 'Delete Row', 'learndash-slack' ),
			'default_title' => __( 'New Row', 'learndash-slack' ),
			'input_name' => false,
		) );
		
		// Ensure Dummy Field is created
		$field_count = ( count( $args['std'] ) >= 1 ) ? count( $args['std'] ) : 1;
		
		$name = $args['input_name'] !== false ? $args['input_name'] : esc_attr( $args['name_attr_wrap'] ) . '[' . esc_attr( $args['id'] ) . ']';
		
		do_action( 'ld_slack_before_repeater' );
		
		?>

		<div data-ld-slack-rbm-repeater class="ld-slack-rbm-repeater <?php echo ( isset( $args['classes'] ) ) ? ' ' . implode( ' ', $args['classes'] ) : ''; ?>">
			
			<div data-repeater-list="<?php echo $name; ?>" class="ld-slack-rbm-repeater-list">

					<?php for ( $index = 0; $index < $field_count; $index++ ) : $value = ( isset( $args['std'][$index] ) ) ? $args['std'][$index] : array(); ?>
				
						<div data-repeater-item<?php echo ( ! isset( $args['std'][$index] ) ) ? ' data-repeater-dummy style="display: none;"' : ''; ?> class="ld-slack-rbm-repeater-item">
							
							<table class="repeater-header wp-list-table widefat fixed posts">

								<thead>

									<tr>
										<th scope="col">
											<div class="title" data-repeater-default-title="<?php echo $args['default_title']; ?>">

												<?php if ( isset( $args['std'][$index] ) && reset( $args['std'][$index] ) !== '' ) : 

													// Surprisingly, this is the most efficient way to do this. http://stackoverflow.com/a/21219594
													foreach ( $value as $key => $setting ) : ?>
														<?php echo $setting; ?>
													<?php 
														break;
													endforeach; 

												else: ?>

													<?php echo $args['default_title']; ?>

												<?php endif; ?>

											</div>
											
											<div class="ld-slack-rbm-repeater-controls">
											
												<input data-repeater-edit type="button" class="button" value="<?php echo $args['edit_item_text']; ?>" />
												<input data-repeater-delete type="button" class="button button-danger" value="<?php echo $args['delete_item_text']; ?>" />
												
											</div>

										</th>

									</tr>

								</thead>

							</table>
							
							<div class="ld-slack-rbm-repeater-content reveal" data-reveal data-v-offset="64">
								
								<div class="ld-slack-rbm-repeater-form">

									<table class="widefat" width="100%" cellpadding="0" cellspacing="0">

										<tbody>

											<?php foreach ( $args['fields'] as $field_id => $field ) : ?>

												<tr>

													<?php if ( is_callable( "ld_slack_{$field['type']}_callback" ) ) : 

														$field['id'] = $field_id;
														$field['std'] = ( isset( $value[ $field_id ] ) ) ? $value[ $field_id ] : $field['std'];

														if ( $field['type'] == 'checkbox' ) : 

															if ( isset( $field['std'] ) && (int) $field['std'] !== 0 ) {
																$field['field_class'][] = 'default-checked';
															}

														endif;

														if ( $field['type'] !== 'hook' ) : ?>

															<td>

																<?php call_user_func( "ld_slack_{$field['type']}_callback", $field ); ?>

															</td>

														<?php else : 

															// Don't wrap calls for a Hook
															call_user_func( "ld_slack_{$field['type']}_callback", $field ); 

														endif;
		
													else :
		
														call_user_func( 'ld_slack_missing_callback', $field );

													endif; ?>

												</tr>

											<?php endforeach; ?>

										</tbody>

									</table>
									
									<input type="submit" class="button button-primary alignright" value="<?php echo $args['save_item_text']; ?>" data-saving_text="<?php echo $args['saving_item_text']; ?>" />
								  
								</div>
								
								<a class="close-button" data-close aria-label="<?php echo _x( 'Close Notification Editor', 'Close Slack Notification Modal', 'learndash-slack' ); ?>">
									<span aria-hidden="true">&times;</span>
								</a>
								
							</div>
							
						</div>

					<?php endfor; ?>	  

			</div>
			
			<input data-repeater-create type="button" class="button" style="margin-top: 6px;" value="<?php echo $args['add_item_text']; ?>" />

		</div>
		
		<?php
		
		do_action( 'ld_slack_after_repeater' );
		
	}
	
	/**
	 * Adds a Tooltip to Fields
	 * 
	 * @param		array  $args Field Args
	 *                            
	 * @access		public
	 * @since		1.1.0
	 * @return		void
	 */
	public function add_tooltip( $args ) {
		
		$args = wp_parse_args( $args, array(
			'tooltip_title' => '',
			'tooltip_desc' => '',
		) );
		
		if ( ! empty( $args['tooltip_title'] ) && ! empty( $args['tooltip_desc'] ) ) {
			$tooltip = '<span alt="f223" class="ld-slack-help-tip dashicons dashicons-editor-help" title="<strong>' . $args['tooltip_title'] . '</strong>: ' . $args['tooltip_desc'] . '"></span>';
			
			echo $tooltip;
			
		}
		
	}
	
	/**
	 * Creating a Hidden Field for a Post ID works out more simply using a Hook. 
	 * 
	 * @param		array  Field Args
	 * 
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function post_id_field( $args ) {
		
		// Post ID of 0 on wp_insert_post() auto-generates an available Post ID
		if ( empty( $args['std'] ) ) $args['std'] = 0;
		?>

		<input type="hidden" class="ld-slack-field ld-slack-post-id" name="<?php echo $args['id']; ?>" value="<?php echo (string) $args['std']; ?>" />

	<?php
	}
	
	/**
	 * Creating a Hidden Field for a Term ID works out more simply using a Hook. 
	 * 
	 * @param		array  Field Args
	 * 
	 * @access		public
	 * @since		1.2.0
	 * @return		void
	 */
	public function term_id_field( $args ) {
		
		// If Term ID is 0, we will be using wp_insert_term() rather than wp_update_term()
		if ( empty( $args['std'] ) ) $args['std'] = 0;
		?>

		<input type="hidden" class="ld-slack-field ld-slack-term-id" name="<?php echo $args['id']; ?>" value="<?php echo (string) $args['std']; ?>" />

	<?php
	}
	
	/**
	 * Create the Replacements Field based on the returned Array
	 * 
	 * @param		array  $args Field Args
	 *						  
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function replacement_hints_field( $args ) {
		
		$args = wp_parse_args( $args, array(
			'std' => false,
		) );
		
		$hints = $this->get_replacement_hints();
		$selected = $args['std'];
		
		foreach ( $hints as $class => $hint ) : ?>

			<td class="ld-slack-replacement-instruction <?php echo $class; ?><?php echo ( $class !== $selected ) ? ' hidden' : ''; ?>">
				<div class="header-text">
					<?php echo _x( 'Here are the available text replacements to use in the Message Pre-Text, Message Title, and Message Fields for the Slack Trigger selected:', 'Text Replacements Label', 'learndash-slack' ); ?>

				</div>

				<?php foreach ( $hint as $replacement => $label ) : ?>

					<div class="replacement-wrapper">
						<span class="replacement"><?php echo $replacement; ?></span> : <span class="label"><?php echo $label; ?></span>
					</div>

				<?php endforeach; ?>

				<?php if ( $class == 'complete_course' ) : ?>

					<div class="additional-help-text <?php echo $class; ?>">
						<?php printf( _x( '<em>Cumulative</em> is the average for all the %s in the %s.', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quizzes' ), LearnDash_Custom_Label::get_label( 'course' ) ); ?>
						<br />
						<?php printf( _x( '<em>Aggregate</em> is the sum for all the %s in the %s.', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quizzes' ), LearnDash_Custom_Label::get_label( 'course' ) ); ?>
					</div>

				<?php endif; ?>
			</td>

		<?php endforeach;
		
	}
	
	/**
	 * Filterable Array holding all the Text Replacement Hints for all the Triggers
	 * 
	 * @access		private
	 * @since		1.0.0
	 * @return		array   Array of Text Replacement Hints for each Trigger
	 */
	private function get_replacement_hints() {

		/**
		* Add extra Replacement Hints directly to the User Group
		*
		* @since 1.0.0
		*/
		$user_hints = apply_filters( 'ld_slack_user_replacement_hints', array(
			'%username%' => _x( 'Display the User\'s username.', '%username% Hint Text', 'learndash-slack' ),
			'%email%' => _x( 'Display the User\'s email.', '%email% Hint Text', 'learndash-slack' ),
			'%name%' => _x( 'Display the User\'s display name.', '%name% Hint Text', 'learndash-slack' ),
		) );
		
		$course_basic_hints = array(
			'%course_name%' => sprintf( _x( 'Display the %s name.', '%course_name% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ) ),
			'%course_link%' => sprintf( _x( 'Show a link to the %s.', '%course_link% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ) ),
		);
		
		$course_point_hints = array();
		
		// If we're on LD >= 2.4, Course Points exist
		if ( version_compare( LEARNDASH_VERSION, '2.4' ) >= 0 ) {
			
			$course_point_hints['%student_course_points%'] = sprintf( _x( 'The total %s Points the Student has.', '%student_course_points% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ) );
			$course_point_hints['%access_course_points%'] = sprintf( _x( 'The number of %s Points required to access the %s.', '%access_course_points% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ), LearnDash_Custom_Label::get_label( 'course' ) );
			
		}
		
		/**
		* Keeps Course Point stuff out of the Basic Hints so that useless Replacement Hints don't end up in Lessons/Topics/etc.
		* 
		* @since 1.2.4
		*/
		$course_point_hints = apply_filters( 'ld_slack_course_point_replacement_hints', $course_point_hints );
		
		/**
		* Add extra Replacement Hints directly to the Basic Course Hint Group
		* 
		* @since 1.0.0
		*/
		$course_basic_hints = apply_filters( 'ld_slack_course_basic_replacement_hints', $course_basic_hints );

		$course_advanced_hints = array(
			'%timestamp%' => sprintf( _x( 'Display the time when %s was completed.', '%timestamp% Hint Text for Courses', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ) ),
			'%cumulative_correct%' => sprintf( _x( 'Display the average points scored across all %s in the %s.', '%cumulative_correct% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quizzes' ), LearnDash_Custom_Label::get_label( 'course' ) ),
			'%cumulative_incorrect%' => sprintf( _x( 'Display the average points missed across all %s in the %s.', '%cumulative_incorrect% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quizzes' ), LearnDash_Custom_Label::get_label( 'course' ) ),
			'%cumulative_percentage%' => sprintf( _x( 'Display the average percentage of correct answers across all %s in the %s.', '%cumulative_percentage% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quizzes' ), LearnDash_Custom_Label::get_label( 'course' ) ),
			'%cumulative_timespent%' => sprintf( _x( 'Display the average time spent across all %s in the %s.', '%cumulative_timespent% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quizzes' ), LearnDash_Custom_Label::get_label( 'course' ) ),
			'%cumulative_total%' => sprintf( _x( 'Display the average of the total number of questions in the %s', '%cumulative_total% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ) ),
			'%aggregate_correct%' => sprintf( _x( 'Display the sum of the points scored across all %s in the %s.', '%aggregate_correct% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quizzes' ), LearnDash_Custom_Label::get_label( 'course' ) ),
			'%aggregate_incorrect%' => sprintf( _x( 'Display the sum of the points missed across all %s in the %s.', '%aggregate_incorrect% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quizzes' ), LearnDash_Custom_Label::get_label( 'course' ) ),
			'%aggregate_percentage%' => sprintf( _x( 'Display the sum of the percentage of correct answers across all %s in the %s.', '%aggregate_percentage% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quizzes' ), LearnDash_Custom_Label::get_label( 'course' ) ),
			'%aggregate_timespent%' => sprintf( _x( 'Display the sum of the time spent across all %s in the %s.', '%aggregate_timespent% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quizzes' ), LearnDash_Custom_Label::get_label( 'course' ) ),
			'%aggregate_total%' => sprintf( _x( 'Display the sum of the total number of questions on the %s', '%aggregate_total% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ) ),
		);
		
		// If we're on LD >= 2.4, Course Points exist
		if ( version_compare( LEARNDASH_VERSION, '2.4' ) >= 0 ) {
			
			$course_advanced_hints = array( '%awarded_course_points%' => sprintf( _x( 'The %s Points awarded upon completion.', '%awarded_course_points% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ) ) ) + $course_advanced_hints;
			
		}
		
		/**
		* Add extra Replacement Hints directly to the Advanced Course Hint Group
		* 
		* @since 1.0.0
		*/
		$course_advanced_hints = apply_filters( 'ld_slack_course_advanced_replacement_hints', $course_advanced_hints );

		/**
		* Add extra Replacement Hints directly to the Lesson Hint Group
		* 
		* @since 1.0.0
		*/
		$lesson_hints = apply_filters( 'ld_slack_lesson_replacement_hints', array(
			'%lesson_name%' => sprintf( _x( 'Display the %s name.', '%lesson_name% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'lesson' ) ),
			'%lesson_link%' => sprintf( _x( 'Show a link to the %s.', '%lesson_link% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'lesson' ) ),
		) );
		
		/**
		* Add extra Replacement Hints directly to the Advanced Lesson Hint Group
		* 
		* @since 1.2.4
		*/
		$lesson_advanced_hints = apply_filters( 'ld_slack_lesson_advanced_replacement_hints', array(
			'%timestamp%' => sprintf( _x( 'Display the time when the %s was completed.', '%timestamp% Hint Text for Lessons', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'lesson' ) ),
		) );

		/**
		* Add extra Replacement Hints directly to the Topic Hint Group
		* 
		* @since 1.0.0
		*/
		$topic_hints = apply_filters( 'ld_slack_topic_replacement_hints', array(
			'%topic_name%' => sprintf( _x( 'Display the %s name.', '%topic_name% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'topic' ) ),
			'%topic_link%' => sprintf( _x( 'Show a link to the %s.', '%topic_link% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'topic' ) ),
			'%timestamp%' => sprintf( _x( 'Display the time when the %s was completed.', '%timestamp% Hint Text for Topics', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'topic' ) ),
		) );
	  
		/**
		* Add extra Replacement Hints directly to the Quiz Hint Group
		* 
		* @since 1.0.0
		*/
		$quiz_hints = apply_filters( 'ld_slack_quiz_replacement_hints', array(
			'%quiz_name%' => sprintf( _x( 'Display the %s name.', '%quiz_name% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quiz' ) ),
			'%quiz_link%' => sprintf( _x( 'Show a link to the %s.', '%quiz_link% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quiz' ) ),
			'%correct%' => sprintf( _x( 'Display the number of correct answers given on the %s', '%correct% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quiz' ) ),
			'%incorrect%' => sprintf( _x( 'Display the number of incorrect answers given on the %s', '%incorrect% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quiz' ) ),
			'%total%' => sprintf( _x( 'Display the total number of questions on the %s', '%total% Hint Text for Quizzes', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quiz' ) ),
			'%percentage%' => sprintf( _x( 'Display the percentage of correct answers on the %s.', '%percentage% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quiz' ) ),
			'%timestamp%' => sprintf( _x( 'Display the time when the %s was completed.', '%timestamp% Hint Text for Quizzes', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quiz' ) ),
			'%time_spent%' => sprintf( _x( 'Display the how long it took to complete the %s.', '%time_spent% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quiz' ) ),
			'%questions_pending_grading%' => sprintf( _x( 'Shows a list of links to Questions pending manual grading for the %s submission.', '%questions_pending_grading% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quiz' ) ),
		) );
		
		/**
		* Add extra Replacement Hints directly to the Essay Hint Group
		* 
		* @since 1.0.0
		*/
		$essay_hints = apply_filters( 'ld_slack_essay_replacement_hints', array(
			'%points_earned%' => _x( 'Display the total points earned.', '%points_earned% Hint Text', 'learndash-slack' ),
			'%points_possible%' => _x( 'Display the total points possible.', '%points_possible% Hint Text', 'learndash-slack' ),
		) );

		/**
		 * Add extra Replacement Hints directly to the Upload Assignment Hint Group
		 * 
		 * @since 1.2.0
		 */
		$upload_assignment_hints = apply_filters( 'ld_slack_upload_assignment_replacement_hints', array(
			'%assignment_name%' => _x( 'Display the Assignment name.', '%assignment_name% Hint Text', 'learndash-slack' ),
			'%file_name%' => _x( 'Display the Assignment file name.', '%file_name% Hint Text', 'learndash-slack' ),
			'%file_link%' => _x( 'Display the Assignment file link.', '%file_link% Hint Text', 'learndash-slack' ),
			'%lesson_type%' => sprintf( _x( 'Display the %s type.', '%lesson_type% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'lesson' ) ),
			'%course_name%' => sprintf( _x( 'Display the %s name.', '%course_name% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ) ),
			'%course_link%' => sprintf( _x( 'Show a link to the %s.', '%course_link% Hint Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ) ),
			'%points_possible%' => _x( 'Display the total points possible.', '%points_possible% Hint Text', 'learndash-slack' ),
		) );
		
		/**
		 * Add extra Replacement Hints directly to the Approve Assignment Hint Group
		 * 
		 * @since 1.2.0
		 */
		$approve_assignment_hints = apply_filters( 'ld_slack_approve_assignment_replacement_hints', $upload_assignment_hints + array(
			'%points_earned%' => _x( 'Display the total points earned.', '%points_earned% Hint Text', 'learndash-slack' ),
		) );
		
		$x_days_hints = apply_filters( 'ld_slack_x_days_replacement_hints', array(
			'%x_days%' => _x( 'The number of days before/after the Event.', '%x_days% Hint Text', 'learndash-slack' ),
		) );

		$replacement_hints = array(
			'enroll_course' => array_merge( $user_hints, $course_basic_hints, $course_point_hints ),
			'complete_course' => array_merge( $user_hints, $course_basic_hints, $course_point_hints, $course_advanced_hints ),
			'complete_lesson' => array_merge( $user_hints, $course_basic_hints, $lesson_hints, $lesson_advanced_hints ),
			'lesson_available' => array_merge( $user_hints, $course_basic_hints, $lesson_hints ),
			'complete_topic' => array_merge( $user_hints, $course_basic_hints, $lesson_hints, $topic_hints ),
			'pass_quiz' => array_merge( $user_hints, $quiz_hints, $course_basic_hints ),
			'fail_quiz' => array_merge( $user_hints, $quiz_hints, $course_basic_hints ),
			'complete_quiz' => array_merge( $user_hints, $quiz_hints, $course_basic_hints ),
			'upload_assignment' => array_merge( $user_hints, $upload_assignment_hints, $lesson_hints ),
			'approve_assignment' => array_merge( $user_hints, $approve_assignment_hints, $lesson_hints ),
			'not_logged_in' => array_merge( $user_hints, $x_days_hints ),
			'course_expires' => array_merge( $course_basic_hints, $course_point_hints, $x_days_hints ),
			'course_enrollment' => array_merge( $user_hints, $course_basic_hints, $course_point_hints ),
			'essay_graded' => array_merge( $user_hints, $course_basic_hints, $lesson_hints, $essay_hints ),
		);
		
		/**
		* Allows additional Triggers to be added to the Replacement Hints
		*
		* @since 1.0.0
		*/
		return apply_filters( 'ld_slack_text_replacement_hints', $replacement_hints, $user_hints, $course_basic_hints );
		
	}
	
	/**
	 * Localize the Admin.js with some values from PHP-land
	 * 
	 * @param	  array $localization Array holding all our Localizations
	 *														
	 * @access	  public
	 * @since	  1.0.0
	 * @return	  array Modified Array
	 */
	public function localize_script( $localization ) {
		
		$localization['i18n'] = array(
			'activeText' => _x( 'Active Notification', 'Active Notification Aria Label', 'learndash-slack' ),
			'inactiveText' => _x( 'Inactive Notification', 'Inactive Notification Aria Label', 'learndash-slack' ),
			'confirmNotificationDeletion' => _x( 'Are you sure you want to delete this Slack Notification?', 'Confirm Notification Deletion', 'learndash-slack' ),
			'confirmWebhookDeletion' => _x( 'Are you sure you want to delete this Slack Webhook?', 'Confirm Webhook Deletion', 'learndash-slack' ),
			'validationError' => _x( 'This field is required', 'Required Field not filled out (Ancient/Bad Browsers Only)', 'learndash-slack' ),
			'onbeforeunload' => __( 'Any unsaved changes will be lost. Are you sure you want to leave this page?', 'learndash-slack' ),
		);
		
		$localization['ajax'] = admin_url( 'admin-ajax.php' );
		
		return $localization;
		
	}
	
	/**
	 * Enqueue our CSS/JS on our Settings Page
	 * 
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function admin_enqueue_scripts() {
		
		global $current_screen;

		if ( $current_screen->base == 'settings_page_learndash_slack' ) {

			wp_enqueue_style( 'wp-color-picker' );
			wp_enqueue_style( 'learndash-slack-admin' );
			
			// Dependencies
			wp_enqueue_script( 'jquery-effects-core' );
			wp_enqueue_script( 'jquery-effects-highlight' );
			wp_enqueue_script( 'jquery-ui-tooltip' );
			wp_enqueue_script( 'jquery-ui-sortable' );
			
			wp_enqueue_script( 'learndash-slack-admin' );
			
		}
		
	}
	
	/**
	 * Fix the Admin Title since our pages "don't exist"
	 * 
	 * @param		string $admin_title The page title, with extra context added
	 * @param		string $title       The original page title
	 *                                               
	 * @access		public
	 * @since		1.2.0
	 * @return		string Admin Title
	 */
	public function admin_title( $admin_title, $title ) {
		
		global $current_screen;
		
		if ( $current_screen->base == 'settings_page_learndash_slack' ) {
			return __( 'LearnDash Slack Settings', 'learndash-slack' ) . $admin_title;
		}
		
		return $admin_title;
		
	}
	
	/**
	 * Add saving indicator to the Timestamp Format Field
	 * Since it is not in a Repeater, this is the quickest/easiest way to do so without damaging other Fields
	 * 
	 * @param		array $args Field Args
	 *                           
	 * @access		public
	 * @since		1.2.0
	 * @return		void
	 */
	public function add_saving_to_timestamp_format( $args ) {
		
		if ( $args['id'] !== 'timestamp_format' ) return;
		
		?>

		<div class="field-group saving" style="display: none;">
			<span class="dashicons dashicons-clock"></span>
		</div>

		<?php 
		
	}

}