<?php
/**
 * Integrates subscriptions into Courses.
 *
 * @since 1.0.0
 */

defined( 'ABSPATH' ) || die();

/**
 * Class LD_ConvertKit_Courses
 *
 * Integrates subscriptions into Courses.
 *
 * @since 1.0.0
 */
class LD_ConvertKit_Courses {

	/**
	 * LD_ConvertKit_Courses constructor.
	 *
	 * @since 1.0.0
	 */
	function __construct() {

		add_action( 'learndash_update_course_access', array( $this, 'course_enroll' ), 10, 4 );
		add_filter( 'ld_after_course_status_template_container', array( $this, 'add_subscription_form_to_course' ) );
		add_action( 'init', array( $this, 'subscribe_form_add_subscriber' ) );
	}

	/**
	 * Subscribes a user to the proper ConvertKit Tag when enrolling in a Course.
	 *
	 * @since 1.1.0
	 * @access private
	 *
	 * @param int $user_ID
	 * @param int $course_ID
	 * @param array $access_list
	 * @param bool $remove True if removing the Course from user, false otherwise.
	 */
	function course_enroll( $user_ID, $course_ID, $access_list, $remove ) {

		if ( ld_convertkit_get_option_field( 'auto_subscribe' ) !== '1' ) {

			return;
		}

		if ( $remove === false ) {

			$this->subscribe_user( $user_ID, $course_ID );

		} else {

			// TODO User course access revoked API actions.
		}
	}

	/**
	 * Custom content after the Course Status section of the Course template output
	 *
	 * @since 1.0.0
	 *
	 * @param string $after_course_status Content to place after the Course Status
	 *
	 * @return string Content to place after the Course Status
	 */
	public function add_subscription_form_to_course( $after_course_status = '' ) {

		// Don't add subscription form if auto-subscribe is enabled
		if ( ld_convertkit_get_option_field( 'auto_subscribe' ) === '1' ) {

			return $after_course_status;
		}

		$course_id = get_the_ID();

		if ( ! self::course_subscribe_form_enabled( $course_id ) ) {

			return $after_course_status;
		}

		$subscriber_form = ld_convertkit_get_option_field( 'subscriber_form' );

		if ( ! $subscriber_form ) {

			return $after_course_status;
		}

		$user_id              = get_current_user_id();
		$user_can_take_course = sfwd_lms_has_access( $course_id, $user_id );

		if ( ! $user_can_take_course ) {

			return $after_course_status;
		}

		ob_start();
		self::show_subscribe_form( $course_id );
		$subscribe_form = ob_get_clean();

		return $after_course_status . $subscribe_form;
	}

	/**
	 * Shows the Course subscription form.
	 *
	 * @since 1.0.0
	 *
	 * @param string|int $course_id
	 *
	 * @return string Form HTML
	 */
	public static function show_subscribe_form( $course_id ) {

		if ( isset( $_REQUEST['ld_subscription_status'] ) ) {

			if ( $_REQUEST['ld_subscription_status'] === 'success' ) {

				ld_convertkit_load_template( 'subscribe-form-success' );

				return;

			} else {

				ld_convertkit_load_template( 'subscribe-form-failed' );

				return;
			}
		}

		if ( LearnDash_ConvertKit()->courses->is_user_subscribed( get_current_user_id(), $course_id ) ) {

			echo '<p class="ld-convertkit-user-subscribed-text">' . ld_convertkit_subscribed_text() . '</p>';

		} else {

			ld_convertkit_subscribe_form( $course_id );
		}
	}

	/**
	 * Determines if the subscribe form for a Course is enabled.
	 *
	 * @since 1.0.0
	 *
	 * @param int|string $course_id
	 *
	 * @return bool
	 */
	public static function course_subscribe_form_enabled( $course_id ) {

		return ld_convertkit_get_field( 'disable_subscription_form', $course_id ) !== '1';
	}

	/**
	 * Subscribes a user to the Form and/or adds the Course Tag to the user subscription.
	 *
	 * @since 1.0.0
	 */
	public function subscribe_form_add_subscriber() {

		if ( ! LD_ConvertKit_API() ) {

			return;
		}

		$course_ID = isset( $_POST['ld_convertkit_course_id'] ) ? $_POST['ld_convertkit_course_id'] : false;

		if ( ! $course_ID ||
		     ! isset( $_POST['ld_convertkit_submit_subscribed'] ) ||
		     ! wp_verify_nonce(
			     $_POST['ld_convertkit_subscribe_course_nonce'],
			     "ld_convertkit_subscribe_course_id_$course_ID"
		     )
		) {

			return;
		}

		$subscriber_ID = $this->subscribe_user( get_current_user_id(), $course_ID );

		wp_safe_redirect( add_query_arg( 'ld_subscription_status', $subscriber_ID !== false ? 'success' : 'failed', $_REQUEST['_wp_http_referrer'] ) );
		exit();
	}

	/**
	 * Subscribes a user to the Form and Tag.
	 *
	 * @since 1.1.0
	 * @access private
	 *
	 * @param string|int $user_ID
	 * @param string|int $course_ID
	 *
	 * @return bool|int Subscriber ID on success, false on failure.
	 */
	public function subscribe_user( $user_ID, $course_ID ) {

		$subscriber_form = ld_convertkit_get_option_field( 'subscriber_form' );

		if ( $user_ID && $subscriber_form ) {

			$user_data = get_userdata( $user_ID );

			$tag_id = ld_convertkit_ensure_resource_tag( $course_ID );

			$subscriber_ID = LD_ConvertKit_API()->form_subscribe( $subscriber_form, array(
				'email'      => $user_data->user_email,
				'first_name' => $user_data->first_name,
				'tags'       => $tag_id,
			) );

			if ( $subscriber_ID !== false ) {

				ld_convertkit_set_user_subscribe_id( $user_ID, $subscriber_ID );
			}

			return (int) $subscriber_ID;

		} else {

			return false;
		}
	}

	/**
	 * Determines if a user is subscribed to a specific Course.
	 *
	 * @since 1.0.0
	 *
	 * @param string|int $user_ID User ID.
	 * @param bool|string|int $course_ID Course ID. Set to false to determine if user is subscribed to form.
	 * @param bool $check_api If true, will ping API to check, otherwise will used cached info.
	 *
	 * @return bool True if subscribed, false otherwise.
	 */
	public function is_user_subscribed( $user_ID, $course_ID = false, $check_api = false ) {

		if ( $check_api === true && ! LD_ConvertKit_API() ) {

			return false;
		}

		// Check cached
		if ( $check_api === false && $course_ID === false ) {

			$subscriber_ID = ld_convertkit_get_user_subscribe_id( $user_ID );

			return (int) $subscriber_ID > 0;
		}

		$user_data       = get_userdata( $user_ID );
		$subscriber_form = ld_convertkit_get_option_field( 'subscriber_form' );

		if ( $course_ID === false ) {

			$subscribers = LD_ConvertKit_API()->get_form_subscriptions( $subscriber_form );
			$emails      = ld_convertkit_get_emails_from_subscriptions_data( $subscribers );

			return in_array( $user_data->user_email, $emails );
		}

		$tag_id = ld_convertkit_ensure_resource_tag( $course_ID );

		if ( $subscriber_form && $tag_id ) {

			$subscriptions_data = LD_ConvertKit_API()->get_form_subscriptions( $subscriber_form );
			$subscribers        = ld_convertkit_get_emails_from_subscriptions_data( $subscriptions_data );

			if ( in_array( $user_data->user_email, $subscribers ) ) {

				// See if this subscriber has this Course as a tag.
				$subscriber      = LD_ConvertKit_API()->get_subscriber_by_email( $user_data->user_email );
				$subscriber_tags = LD_ConvertKit_API()->get_subscriber_tags( $subscriber['id'] );

				$tag = ld_convertkit_search_array_by_key( $subscriber_tags, 'id', $tag_id );

				return $tag !== false;
			}
		}

		return false;
	}
}