<?php

namespace uncanny_learndash_groups;

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Class AdminMenu
 * @package uncanny_learndash_groups
 */
class RestApiEndPoints {


	/**
	 * The name of the plugin
	 *     *
	 * @since    1.0.0
	 * @access   private
	 * @var      string
	 */
	private $root_path = 'ulgm_management/v1/';

	/**
	 * class constructor
	 */
	public function __construct() {

		if ( defined( 'WP_DEBUG' ) ) {
			if ( false === WP_DEBUG ) {
				error_reporting( 0 );
			}
		}

		//register api class
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );

	}

	/**
	 * Rest API Custom Endpoints
	 *
	 * @since 1.0
	 */
	function register_routes() {

		register_rest_route( $this->root_path, '/add_user/', array(
			'methods'  => 'POST',
			'callback' => array( $this, 'add_user' )
		) );

		register_rest_route( $this->root_path, '/add_group_leader/', array(
			'methods'  => 'POST',
			'callback' => array( $this, 'add_group_leader' )
		) );

		register_rest_route( $this->root_path, '/remove_group_leaders/', array(
			'methods'  => 'POST',
			'callback' => array( $this, 'remove_group_leaders' )
		) );

		register_rest_route( $this->root_path, '/save_general_settings/', array(
			'methods'  => 'POST',
			'callback' => array( $this, 'save_general_settings' )
		) );

		register_rest_route( $this->root_path, '/save_email_templates/', array(
			'methods'  => 'POST',
			'callback' => array( $this, 'save_email_templates' )
		) );

		register_rest_route( $this->root_path, '/upload_users/', array(
			'methods'  => 'POST',
			'callback' => array( $this, 'upload_users' )
		) );

		register_rest_route( $this->root_path, '/email_users/', array(
			'methods'  => 'POST',
			'callback' => array( $this, 'email_users' )
		) );

		register_rest_route( $this->root_path, '/remove_users/', array(
			'methods'  => 'POST',
			'callback' => array( $this, 'remove_users' )
		) );

//		register_rest_route( $this->root_path, '/resend_invitations/', array(
//			'methods'  => 'POST',
//			'callback' => array( $this, 'resend_invitations' )
//		) );

		register_rest_route( $this->root_path, '/download_keys_csv/', array(
			'methods'  => 'POST',
			'callback' => array( $this, 'download_keys_csv' )
		) );

//		register_rest_route( $this->root_path, '/send_test_email_templates/', array(
//			'methods'  => 'POST',
//			'callback' => array( $this, 'send_test_email_templates' )
//		) );

	}

	/**
	 * Adds a user automatically or sends out a redemption code
	 *
	 * @since 1.0
	 */
	function add_user() {
		//Utilities::log( $_POST, '$_POST', true, 'post' );
		// Actions permitted by the pi call (colleced from input element with name action )
		$permitted_actions = array( 'add-invite', 'send-enrollment', 'add-existing-user' );

		// Was an action received, and is the actions allowed
		if ( isset( $_POST['action'] ) && in_array( $_POST['action'], $permitted_actions ) ) {

			$action = (string) $_POST['action'];

		} else {
			$action          = '';
			$data['message'] = __( 'Select an action.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		// Does the current user have permission
		$permission = apply_filters( 'group_management_add_user_permission', 'group_leader' );
		if ( ! current_user_can( $permission ) && ! current_user_can( 'upload_plugins' ) ) {
			$data['message'] = __( 'You do not have permission to add users.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		// If a single part of data doesn't validate, it dies in the validate_add_user_data function and sends back the validation error
		$user_data = $this->validate_new_user_data( $action );

		// Add user and send out welcome email
		if ( 'add-invite' === $action ) {

			$email = $user_data['user_email'];

			if ( email_exists( $email ) || username_exists( $email ) ) {
				// Add existing user and send out welcome email
				$user_id              = email_exists( $email );
				$user_data['user_id'] = $user_id;
				$this->add_existing_user( $user_data );

			} else {
				// Add the user and send out a welcome email
				$this->add_invite_user( $user_data, false );
			}
		}

		// Send Redemption email
		if ( 'send-enrollment' === $action ) {

			// User invitation email subject
			$ulgm_invitation_user_email_subject = SharedVariables::user_invitation_email_subject();

			// User invitation email body
			$ulgm_invitation_user_email_body = SharedVariables::user_invitation_email_body();
			// Add user to invitation list and send out email
			$this->send_redemption_email( $user_data, $ulgm_invitation_user_email_subject, $ulgm_invitation_user_email_body );


		}

	}

	/**
	 * Adds a user automatically or sends out a redemption code
	 *
	 * @since 1.0
	 */
	function upload_users() {

		// Actions permitted by the pi call (colleced from input element with name action )
		$permitted_actions = array( 'add-invite', 'send-enrollment' );

		// Was an action received, and is the actions allowed
		if ( isset( $_POST['action'] ) && in_array( $_POST['action'], $permitted_actions ) ) {

			$action = (string) $_POST['action'];

		} else {
			$action          = '';
			$data['message'] = __( 'Select an action.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}


		// Does the current user have permission
		$permission = apply_filters( 'group_management_add_user_permission', 'group_leader' );
		if ( ! current_user_can( $permission ) && ! current_user_can( 'upload_plugins' ) ) {
			$data['message'] = __( 'You do not have permission to add users.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		$do_not_send_emails = false;
		// Was do not send emails received and was it checked off
		if ( isset( $_POST['not-send-emails'] ) && 'not-send-emails' === $_POST['not-send-emails'] ) {

			$do_not_send_emails = true;

		}

		// If a single part of data doesn't validate, it dies in the validate_add_user_data function and sends back the validation error
		$user_data = $this->validate_new_users_data();

		if ( isset( $_POST['group-id'] ) ) {
			$remaining_seats = SharedFunctions::remaining_seats( $_POST['group-id'] );
			if ( count( $user_data ) > $remaining_seats ) {
				$seats           = strtolower( get_option( 'ulgm_per_seat_text_plural', __( 'Seats', 'uncanny-learndash-groups' ) ) );
				$data['message'] = sprintf( __(
					'The number of users in the .csv exceeds the available %s. No users were imported.', 'uncanny-learndash-groups' ),
					$seats );
				wp_send_json_error( $data );
			}
		}


		// Add user and send out welcome email
		if ( 'add-invite' === $action ) {

			$results = array();

			$error_occurred = false;
			// Add the user and send out a welcome email
			foreach ( $user_data as $row_num => $user ) {
				$success = $this->add_invite_user( $user, $do_not_send_emails, true );

				if ( isset( $success['error-code'] ) ) {
					$error_occurred                 = true;
					$results[ $row_num ]['type']    = 'error';
					$results[ $row_num ]['message'] = $success['error-code'];
					$results[ $row_num ]['email']   = $user['user_email'];
				}

				if ( isset( $success['success-code'] ) ) {
					$results[ $row_num ]['type']    = 'success';
					$results[ $row_num ]['message'] = $success['success-code'];
				}
			}
			//Utilities::log( $results, '$results', true, 'csv' );

			if ( $error_occurred ) {

				$data['message'] = __( 'CSV uploaded and executed successfully. Some row(s) were skipped with the following errors:<br>', 'uncanny-learndash-groups' );

				$data['message'] .= '<ul class="error-message">';
				foreach ( $results as $key => $result ) {

					if ( 'error' === $result['type'] ) {
						$row             = $key + 2;
						$email           = $result['email'];
						$data['message'] .= "<li>" . sprintf( $result['message'], $row, $email ) . "</li>";
					}

				}

				$data['message'] .= '</ul>';
				wp_send_json_success( $data );

			} else {
				$data['message'] = __( 'CSV uploaded and executed successfully.', 'uncanny-learndash-groups' );
				//Hook for LD Notification -- send notification for each course in group
				$group_id                     = (int) $_POST['group-id'];
				$do_ld_group_postdata_updated = apply_filters( 'do_ld_group_postdata_filter', false, $group_id );
				if ( $do_ld_group_postdata_updated ) {
					$group_leaders = learndash_get_groups_administrator_ids( $group_id );
					$group_users   = learndash_get_groups_user_ids( $group_id );
					$group_courses = learndash_get_groups_courses_ids( $group_id );
					do_action( 'ld_group_postdata_updated', $group_id, $group_leaders, $group_users, $group_courses );
				}

				wp_send_json_success( $data );
			}


		}

		// Send Redemption email
		if ( 'send-enrollment' === $action ) {

			$results        = array();
			$error_occurred = false;

			// User invitation email subject
			$ulgm_invitation_user_email_subject = SharedVariables::user_invitation_email_subject();

			// User invitation email body
			$ulgm_invitation_user_email_body = SharedVariables::user_invitation_email_body();

			// Add user to invitation list and send out email
			foreach ( $user_data as $row_num => $user ) {

				$success = $this->send_redemption_email( $user, $ulgm_invitation_user_email_subject, $ulgm_invitation_user_email_body, $do_not_send_emails, true );

				if ( isset( $success['error-code'] ) ) {
					$error_occurred                 = true;
					$results[ $row_num ]['type']    = 'error';
					$results[ $row_num ]['message'] = $success['error-code'];
					$results[ $row_num ]['email']   = $user['user_email'];
				}

				if ( isset( $success['success-code'] ) ) {
					$results[ $row_num ]['type']    = 'success';
					$results[ $row_num ]['message'] = $success['success-code'];
				}
			}

			if ( $error_occurred ) {

				$data['message'] = __( 'CSV uploaded and executed successfully. Some row(s) were skipped with the following errors:<br>', 'uncanny-learndash-groups' );

				$data['message'] .= '<ul class="error-message">';
				foreach ( $results as $key => $result ) {

					if ( 'error' === $result['type'] ) {
						$row             = $key + 2;
						$email           = $result['email'];
						$data['message'] .= "<li>" . sprintf( $result['message'], $row, $email ) . "</li>";
					}

				}

				$data['message'] .= '</ul>';
				wp_send_json_success( $data );

			} else {
				$data['message'] = __( 'CSV uploaded and executed successfully.', 'uncanny-learndash-groups' );
				wp_send_json_success( $data );
			}

		}

	}

	/**
	 * sends out emails to all group users
	 *
	 * @since 2.6
	 */
	function email_users() {
		if ( ( isset( $_POST['group_email_nonce'] ) ) && ( isset( $_POST['group_email_group_id'] ) ) && ( ! empty( $_POST['group_email_group_id'] ) ) ) {

			if ( ! is_user_logged_in() ) {
				exit;
			}
			$current_user = wp_get_current_user();
			if ( ( ! learndash_is_group_leader_user( $current_user->ID ) ) && ( ! learndash_is_admin_user( $current_user->ID ) ) ) {
				wp_send_json_error(
					array(
						'message' => __( 'Error: You are not allowed to send emails.', 'uncanny-learndash-groups' )
					)
				);
			}

			$group_email_data = array();

			if ( ( ! isset( $_POST['group_email_group_id'] ) ) || ( empty( $_POST['group_email_group_id'] ) ) ) {
				wp_send_json_error(
					array(
						'message' => __( 'Error: Invalid group.', 'uncanny-learndash-groups' )
					)
				);
			}

			$group_email_data['statuses'] = isset( $_POST['group_email_status'] ) ? $_POST['group_email_status'] : array();

			if ( ( ! isset( $_POST['group_email_status'] ) ) || ( empty( $_POST['group_email_status'] ) ) ) {
				wp_send_json_error(
					array(
						'message' => __( 'Error: No Status Selected.', 'uncanny-learndash-groups' )
					)
				);
			}

			$group_email_data['group_id'] = intval( $_POST['group_email_group_id'] );

			if ( ( ! isset( $_POST['group_email_sub'] ) ) || ( empty( $_POST['group_email_sub'] ) ) ) {
				wp_send_json_error(
					array(
						'message' => __( 'Error: Email subject required.', 'uncanny-learndash-groups' )
					)
				);
			}

			$group_email_data['email_subject'] = wp_strip_all_tags( stripcslashes( $_POST['group_email_sub'] ) );

			if ( ( ! isset( $_POST['group_email_text'] ) ) || ( empty( $_POST['group_email_text'] ) ) ) {
				wp_send_json_error(
					array(
						'message' => __( 'Error: Email message required.', 'uncanny-learndash-groups' )
					)
				);
			}

			$group_email_data['email_message'] = wpautop( stripcslashes( $_POST['group_email_text'] ) );

			if ( ( ! isset( $_POST['group_email_nonce'] ) ) || ( empty( $_POST['group_email_nonce'] ) )
			     || ( ! wp_verify_nonce( $_POST['group_email_nonce'], 'group_email_nonce_' . $group_email_data['group_id'] . '_' . $current_user->ID ) ) ) {
				wp_send_json_error(
					array(
						'message' => __( 'Error: Please refresh page and try again.', 'uncanny-learndash-groups' )
					)
				);
			}

			$group_admin_ids = learndash_get_groups_administrator_ids( $group_email_data['group_id'] );
			if ( in_array( $current_user->ID, $group_admin_ids ) === false ) {
				wp_send_json_error(
					array(
						'message' => __( 'Error: You are not allowed to send emails.', 'uncanny-learndash-groups' )
					)
				);
			}
			$ulgm_email_from = ! empty( get_option( 'ulgm_email_from', '' ) ) ? get_option( 'ulgm_email_from', '' ) : get_bloginfo( 'admin_email' );

			$mail_args = array(
				'to'          => sanitize_email( $current_user->user_email ),
				'subject'     => $group_email_data['email_subject'],
				'message'     => wpautop( $group_email_data['email_message'] ),
				'attachments' => '',
				'headers'     => array(
					//'MIME-Version: 1.0',
					//'Content-Type: text/html',
					'From: ' . $current_user->display_name . ' <' . $ulgm_email_from . '>',
					'Reply-to: ' . $current_user->display_name . ' <' . $current_user->user_email . '>'
				)
			);

			$group_leader_details['from_name']   = $current_user->display_name;
			$group_leader_details ['from_email'] = $ulgm_email_from;
			$group_leader_details ['reply_to']   = $current_user->user_email;

			if ( ! empty( $group_email_data['statuses'] ) ) {
				$group_user_ids                   = learndash_get_groups_user_ids( $group_email_data['group_id'] );
				$learndash_group_enrolled_courses = learndash_group_enrolled_courses( $group_email_data['group_id'] );
				if ( ! empty( $group_user_ids ) ) {
					$email_addresses = array();

					foreach ( $group_user_ids as $user_id ) {
						$user = get_user_by( 'id', $user_id );

						if ( $user ) {
							//$course_progress = get_user_meta( $user->ID, '_sfwd-course_progress', true );
							// If there is not progress than they haven't started anything
							//if ( '' === $course_progress ) {
							//	$status = 'not-started';
							//} else {

							// Default is not completed
							$completed = false;

							// Default progress
							$in_progress = false;

							// Check group progress courses
							foreach ( $learndash_group_enrolled_courses as $course_id ) {
								$course_progress = learndash_course_progress(
									array(
										'course_id' => $course_id,
										'user_id'   => $user->ID,
										'array'     => true,
									)
								);
								// does the groups course and user progress
								if ( empty( $course_progress ) || ! is_array( $course_progress ) || ( is_array( $course_progress ) && 0 === (int) $course_progress['completed'] && 0 === (int) $course_progress['percentage'] ) ) {
									$in_progress = false;
									$completed   = false;
								} elseif ( is_array( $course_progress ) && (int) $course_progress['completed'] === (int) $course_progress['total'] ) {
									$completed = true;
								} elseif ( is_array( $course_progress ) && (int) $course_progress['total'] !== (int) $course_progress['completed'] ) {
									$in_progress = true;
									$completed   = false;
									break;
								}
							}

							// Set Status
							if ( $completed ) {
								$status = 'completed';
							} elseif ( $in_progress ) {
								$status = 'in-progress';
							} else {
								$status = 'not-started';
							}
							//}

							// removing from bcc.
							//$email_addresses[] = 'Bcc: ' . sanitize_email( $user->user_email );
							if ( in_array( $status, $group_email_data['statuses'] ) ) {
								$email_addresses[] = sanitize_email( $user->user_email );
							}
						}
					}

					$group_email_error = null;
					add_action( 'wp_mail_failed', function ( $mail_error ) {
						global $group_email_error;
						$group_email_error = $mail_error;
					} );

					$mail_errors   = [];
					$mail_success  = [];
					$backup_emails = $email_addresses;

					if ( $email_addresses ) {
						// removing from bcc.
						//$mail_args['headers'] = array_merge( $mail_args['headers'], $email_addresses );

						$mail_args = apply_filters( 'ld_group_email_users_args', $mail_args );
						if ( ! empty( $mail_args ) ) {
							do_action( 'ld_group_email_users_before', $mail_args );

							foreach ( $email_addresses as $k => $email_address ) {

								$send_mail = apply_filters( 'ulgm_maybe_send_group_email', true, $email_address, $mail_args['subject'] );
								if ( $send_mail ) {
									$mail_ret        = wp_mail( $email_address,
										$mail_args['subject'],
										$mail_args['message'],
										$this->get_headers( true, $group_leader_details ) );
									$mail_args['to'] = $email_address;
									do_action( 'ld_group_email_users_after', $mail_args, $mail_ret );

									unset( $backup_emails[ $k ] );

									if ( ! $mail_ret ) {
										$mail_errors[] = $email_address;
									} else {
										$mail_success[] = $email_address;
									}
								}
							}

							if ( ! empty( $mail_errors ) ) {
								wp_send_json_error(
									array(
										'message' => sprintf( __( 'Error: Email(s) to %s not sent. Please try again or check with your hosting provider.', 'uncanny-learndash-groups' ), join( ', ', $mail_errors ) )
									)
								);
							}

							if ( ! empty( $mail_success ) ) {
								wp_send_json_success(
									array(
										'message' => sprintf( __( 'Success: Email sent to %d group users.', 'uncanny-learndash-groups' ), count( $mail_success ) )
									)
								);
							}
						} else {
							wp_send_json_error(
								array(
									'message' => __( 'Mail Args empty. Unepected condition from filter: ld_group_email_users_args.', 'uncanny-learndash-groups' )
								)
							);
						}
					} else {
						wp_send_json_error(
							array(
								'message' => esc_html__( 'No users in group.', 'uncanny-learndash-groups' )
							)
						);
					}
				}
				wp_send_json_error(
					array(
						'message' => esc_html__( 'No users in group.', 'uncanny-learndash-groups' )
					)
				);
			} else {
				wp_send_json_error(
					array(
						'message' => esc_html__( 'No status selected.', 'uncanny-learndash-groups' )
					)
				);
			}
		}
		wp_send_json_error();
	}

	/**
	 * @param      $user_data
	 * @param bool $do_not_send_emails
	 * @param bool $counter On error, return(false) or count errors(true)
	 * @param bool $is_api
	 *
	 * @return array success code || error code
	 */
	function add_invite_user( $user_data, $do_not_send_emails = false, $counter = false, $is_api = true ) {


		// Validate that the user does not exist
		if ( email_exists( $user_data['user_email'] ) || username_exists( $user_data['user_email'] ) ) {
			$user_id              = email_exists( $user_data['user_email'] );
			$user_data['user_id'] = $user_id;

			return $this->add_existing_user( $user_data, $do_not_send_emails, 0, 0, 'not-redeemed', $is_api, false, $counter );
		}

		if ( key_exists( 'user_pass', $user_data ) && ! empty( $user_data['user_pass'] ) ) {
			$plain_password = $user_data['user_pass'];
		} else {
			// Create a password
			$plain_password         = wp_generate_password( $length = 10, $include_standard_special_chars = false );
			$user_data['user_pass'] = $plain_password;
		}
		// hash the password before insertion!
		$user_data['role'] = apply_filters( 'uo-groups-user-role', get_option( 'default_role', 'subscriber' ) );


		// Remove new user notifications
		if ( ! function_exists( 'wp_new_user_notification' ) ) {
			function wp_new_user_notification() {
			}
		}

		// add user to group. NOTE group id has already been validate and the script would return error so its safe :)
		if ( key_exists( 'group_id', $user_data ) ) {
			$group_id = (int) $user_data['group_id'];
		} else {
			$group_id = $_POST['group-id'];
		}

		// Get a code
		$code = SharedFunctions::get_sign_up_code_from_group_id( $group_id );

		$group_name = get_the_title( (int) $group_id );

		// No Codes Left
		if ( ! $code ) {

			if ( ! $counter ) {
				$data['message'] = sprintf( __( 'There are no available %s left.', 'uncanny-learndash-groups' ), strtolower( get_option( 'ulgm_per_seat_text_plural', 'Seats' ) ) );
				wp_send_json_error( $data );
			} else {

				return array(
					'error-code' => sprintf(
						__( 'The user on row %s was not imported because there were no available %s left.', 'uncanny-learndash-groups' ),
						'%s', // this is parsed in another function and value should now be translated
						strtolower( get_option( 'ulgm_per_seat_text_plural', 'Seats' ) )
					)
				);
			}

		}

		// Create the user
		$user_id = wp_insert_user( $user_data );

		// User creation failed
		if ( is_wp_error( $user_id ) ) {

			if ( ! $counter ) {
				$data['message'] = __( 'The user could not be added.  Please contact the website administrator.', 'uncanny-learndash-groups' );
				if ( $is_api ) {
					wp_send_json_error( $data );
				}
			} else {
				return array(
					'error-code' => __( 'The user on row %s could not be added.  Please contact the website administrator.', 'uncanny-learndash-groups' )
				);
			}


		}

		if ( isset( $user_data['first_name'] ) ) {
			update_user_meta( $user_id, 'first_name', $user_data['first_name'] );
		}

		if ( isset( $user_data['last_name'] ) ) {
			update_user_meta( $user_id, 'last_name', $user_data['last_name'] );
		}

		// Add new user to a group
		$user_groups = array( $group_id );

		// get grops leaders that are already in the group
		$current_user_groups = learndash_get_users_group_ids( $user_id );

		// Join new group leader and existing groups leaders
		foreach ( $current_user_groups as $current_user_group ) {
			$user_groups[] = $current_user_group->ID;
		}

		// set all group users.
		learndash_set_users_group_ids( $user_id, $user_groups );

		$order_id       = SharedFunctions::get_order_id_from_group_id( $group_id );
		$codes_group_id = SharedFunctions::get_codes_group_id_from_code( $code );
		$set_code       = SharedFunctions::set_sign_up_code_status( $code, $user_id, $order_id, 'not redeemed', $codes_group_id );

		if ( ! $set_code ) {

			if ( ! $counter ) {
				$data['message'] = __( 'The key failed to be assigned.  Please contact the website administrator.', 'uncanny-learndash-groups' );
				if ( $is_api ) {
					wp_send_json_error( $data );
				}
			} else {
				return array(
					'error-code' => __( 'The key failed to be assigned on row %s.  Please contact the website administrator.', 'uncanny-learndash-groups' )
				);
			}

		}

		do_action( 'ulgm_group_user_invited', $user_data, $group_id, $order_id );
		$do_not_send_emails = apply_filters( 'ulgm_do_not_send_new_user_email', $do_not_send_emails, $user_data, $group_id, $order_id );

		if ( $do_not_send_emails ) {

			if ( ! $counter ) {
				$data['message'] = __( 'User has been added.', 'uncanny-learndash-groups' );
				SharedFunctions::delete_transient( null, $group_id );
				if ( $is_api ) {
					wp_send_json_success( $data );
				}
			} else {
				return [ 'success-code' => 'user-added-no-email' ];
			}

		} else {

			// User welcome email subject
			$ulgm_user_welcome_email_subject = SharedVariables::user_welcome_email_subject();

			// User welcome email subject
			$ulgm_user_welcome_email_body = SharedVariables::user_welcome_email_body();

			// Send Welcome email, for extra validation we are sending in the user id and getting user data from WP because there may be filters
			$welcome_email = $this->send_welcome_email( $user_id, $plain_password, $ulgm_user_welcome_email_subject, $ulgm_user_welcome_email_body, $group_name );

			// Welcome Email Failed
			if ( is_wp_error( $welcome_email ) ) {

				if ( ! $counter ) {
					$data['message'] = __( 'User has been added. Welcome email FAILED to send.', 'uncanny-learndash-groups' );
					SharedFunctions::delete_transient( null, $group_id );
					if ( $is_api ) {
						wp_send_json_success( $data );
					}
				} else {
					return array(
						'error-code' => __( 'The invitation email could not be sent to user on row %s.  Please contact the website administrator.', 'uncanny-learndash-groups' )
					);
				}

			}

			if ( ! $counter ) {
				$data['message'] = __( 'User has been added and welcome email is sent.', 'uncanny-learndash-groups' );
				SharedFunctions::delete_transient( null, $group_id );

				//Hook for LD Notification -- send notification for each course in group
				$do_ld_group_postdata_updated = apply_filters( 'do_ld_group_postdata_filter', false, $group_id );
				if ( $do_ld_group_postdata_updated ) {
					$group_leaders = learndash_get_groups_administrator_ids( $group_id );
					$group_users   = learndash_get_groups_user_ids( $group_id );
					$group_courses = learndash_get_groups_courses_ids( $group_id );
					do_action( 'ld_group_postdata_updated', $group_id, $group_leaders, $group_users, $group_courses );
				}

				if ( $is_api ) {
					wp_send_json_success( $data );
				}
			} else {
				return array( 'success-code' => 'user-added-email-sent' );
			}
		}
	}

	/**
	 * @param        $user_data
	 * @param bool $do_not_send_emails
	 * @param int $group_id
	 * @param int $order_id
	 * @param string $code_status
	 * @param bool $is_api
	 * @param bool $is_cron
	 * @param bool $counter
	 *
	 * @return array
	 */
	function add_existing_user( $user_data, $do_not_send_emails = false, $group_id = 0, $order_id = 0, $code_status = 'not redeemed', $is_api = true, $is_cron = false, $counter = false ) {

		if ( ! key_exists( 'user_id', $user_data ) ) {
			$user = get_user_by( 'email', $user_data['user_email'] );

			// Validate that the user does not exist
			if ( ! $user ) {

				if ( $counter ) {
					return array(
						'error-code' => __( 'The user on row %s was not imported. Contact Administration.', 'uncanny-learndash-groups' )
					);
				}

				// User already exists
				$data['message'] = __( 'A user with the email does not exist.', 'uncanny-learndash-groups' );

				if ( $is_api ) {
					wp_send_json_error( $data );
				}
			}

			$user_id = $user->ID;
		} else {
			$user_id = $user_data['user_id'];
			$user    = new \WP_User( $user_id );
		}

		// add user to group. NOTE group id has already been validate and the script would return error so its safe :)
		if ( 0 === absint( $group_id ) ) {
			$group_id = $_POST['group-id'];
		}


		// Add new user to a group
		$user_groups = array( $group_id );

		// get groups leaders that are already in the group
		$current_user_groups = learndash_get_users_group_ids( $user_id );

		if ( ! empty( $current_user_groups ) ) {
			// Join new users
			foreach ( $current_user_groups as $current_user_group ) {
				if ( is_object( $current_user_group ) && $current_user_group->ID == $group_id ) {
					$data['message'] = __( 'This user is already a member of this group.', 'uncanny-learndash-groups' );

					if ( $counter ) {
						return array(
							'error-code' => __( 'The user on row %s is already a member of this group.', 'uncanny-learndash-groups' )
						);
					}

					if ( $is_api ) {
						wp_send_json_error( $data );
					}

				} elseif ( $current_user_group === $group_id ) {

					$data['message'] = __( 'This user is already a member of this group.', 'uncanny-learndash-groups' );

					if ( $counter ) {
						return array(
							'error-code' => __( 'The user on row %s is already a member of this group.', 'uncanny-learndash-groups' )
						);
					}

					if ( $is_api ) {
						wp_send_json_error( $data );
					}
				}
			}
		}

		if ( ! empty( $current_user_groups ) ) {
			// Join new users
			foreach ( $current_user_groups as $current_user_group ) {
				if ( is_object( $current_user_group ) ) {
					$user_groups[] = $current_user_group->ID;
				} else {
					$user_groups[] = $current_user_group;
				}
			}
		}

		// Get a code
		//if ( 'yes' === get_option( 'group_leaders_dont_use_seats', 'no' ) && in_array( 'group_leader', $user->roles ) ) {
		//Do nothing
		//} else {
		$code = SharedFunctions::get_sign_up_code_from_group_id( $group_id, 1, $user_id, $order_id, $is_cron );

		// No Codes Left
		if ( ! $code ) {
			$data['message'] = __( 'There are no codes left.', 'uncanny-learndash-groups' );

			if ( $counter ) {
				return array(
					'error-code' => sprintf(
						__( 'The user on row %s was not imported because there were no available %s left.', 'uncanny-learndash-groups' ),
						'%s', // this is parsed in another function and value should now be translated
						strtolower( get_option( 'ulgm_per_seat_text_plural', 'Seats' ) )
					)
				);
			}

			if ( $is_api ) {
				wp_send_json_error( $data );
			}
		}
		//}

		//if ( 'yes' === get_option( 'do_not_add_group_leader_as_member', 'no' ) && in_array( 'group_leader', $user->roles ) ) {
		//Do nothing
		//} else {
		// set all group users.
		learndash_set_users_group_ids( $user_id, $user_groups );
		// Make sure the user has the group membership role (default: 'subscriber')
		$user->add_role( apply_filters( 'uo-groups-user-role', get_option( 'default_role', 'subscriber' ) ) );
		//}

		if ( 0 === absint( $order_id ) ) {
			$order_id = SharedFunctions::get_order_id_from_group_id( $group_id );
		}
		//Utilities::log( $order_id, '$order_id', true, 'api' );

		//$order_id       = SharedFunctions::get_order_id_from_group_id( $group_id );
		//$codes_group_id = SharedFunctions::get_codes_group_id_from_code( $order_id, $user_id );
		//if ( 'yes' === get_option( 'group_leaders_dont_use_seats', 'no' ) && in_array( 'group_leader', $user->roles ) ) {
		//Do nothing
		//} else {
		$codes_group_id = SharedFunctions::get_codes_group_id_from_code( $code );
		$set_code       = SharedFunctions::set_sign_up_code_status( $code, $user_id, $order_id, $code_status, $codes_group_id );
		$data           = array();
		if ( ! $set_code ) {
			$data['message'] = __( 'Code failed.', 'uncanny-learndash-groups' );

			if ( $counter ) {
				return array(
					'error-code' => __( 'The key failed to be assigned on row %s.  Please contact the website administrator.', 'uncanny-learndash-groups' )
				);
			}

			if ( $is_api ) {
				wp_send_json_error( $data );
			}
		}
		//}

		do_action( 'ulgm_existing_group_user_added', $user_data, $group_id, $order_id );
		$do_not_send_emails = apply_filters( 'ulgm_do_not_send_existing_user_email', $do_not_send_emails, $user_data, $group_id, $order_id );

		if ( $do_not_send_emails ) {

			$data['message'] = __( 'The specified user was already registered on the site and has been automatically added to this group.', 'uncanny-learndash-groups' );

			if ( $counter ) {
				return array(
					'success-code' => 'user-added'
				);
			}
			if ( $is_api ) {
				SharedFunctions::delete_transient( null, $group_id );
				wp_send_json_success( $data );
			}


		} else {

			// User welcome email subject
			$ulgm_user_welcome_email_subject = SharedVariables::exiting_user_welcome_email_subject();

			// User welcome email subject
			$ulgm_user_welcome_email_body = SharedVariables::exiting_user_welcome_email_body();

			// Get group name
			$group_name = get_the_title( $group_id );

			// Send Welcome email, for extra validation we are sending in the user id and getting user data from WP because there may be filters
			$welcome_email = $this->send_welcome_email( $user_id, '', $ulgm_user_welcome_email_subject, $ulgm_user_welcome_email_body, $group_name );

			// Welcome Email Failed
			if ( is_wp_error( $welcome_email ) ) {
				$data['message'] = __( 'User has been added. Welcome email FAILED to send.', 'uncanny-learndash-groups' );

				if ( $counter ) {
					return array(
						'error-code' => __( 'The invitation email could not be sent to user on row %s.  Please contact the website administrator.', 'uncanny-learndash-groups' )
					);
				}

				if ( $is_api ) {
					SharedFunctions::delete_transient( null, $group_id );
					wp_send_json_success( $data );
				}
			}

			$data['message'] = __( 'The specified user was already registered on the site and has been automatically added to this group.', 'uncanny-learndash-groups' );

			if ( $counter ) {
				return array(
					'success-code' => 'user-added-email-sent'
				);
			}

			if ( $is_api ) {
				SharedFunctions::delete_transient( null, $group_id );

				//Hook for LD Notification -- send notification for each course in group
				$do_ld_group_postdata_updated = apply_filters( 'do_ld_group_postdata_filter', false, $group_id );
				if ( $do_ld_group_postdata_updated ) {
					$group_leaders = learndash_get_groups_administrator_ids( $group_id );
					$group_users   = learndash_get_groups_user_ids( $group_id );
					$group_courses = learndash_get_groups_courses_ids( $group_id );
					do_action( 'ld_group_postdata_updated', $group_id, $group_leaders, $group_users, $group_courses );
				}

				wp_send_json_success( $data );
			}

		}
	}

	/**
	 * @param $action
	 *
	 * @return array
	 */
	private function validate_new_user_data( $action ) {

		// Defaults, we don't always need first and last
		$first_name = '';
		$last_name  = '';
		$group_name = '';

		// Was group id received
		if ( isset( $_POST['group-id'] ) ) {

			// is group a valid integer
			if ( ! absint( $_POST['group-id'] ) ) {
				$data['message'] = __( 'Group ID must be a whole number.', 'uncanny-learndash-groups' );
				$data['error']   = 'invalid-group-id';
				wp_send_json_error( $data );
			}

			$group_leader_id = get_current_user_id();
			$user_group_ids  = learndash_get_administrators_group_ids( $group_leader_id, true );

			// is the current user able to administer this group
			if ( ! in_array( $_POST['group-id'], $user_group_ids ) ) {
				$data['message'] = __( 'You do not have permission to manage this group.', 'uncanny-learndash-groups' );
				$data['error']   = 'invalid-group-id';
				wp_send_json_error( $data );
			}

			$group_name = get_the_title( absint( $_POST['group-id'] ) );

		} else {
			$data['message'] = __( 'Group ID was not received. Reload page and try again.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		// Was first name received
		if ( isset( $_POST['first_name'] ) && ( 'add-invite' === $action || 'add-leader' === $action ) ) {

			// is first name a valid name, check if it contains letters only, allow hyphens
			if ( 0 === preg_match( "/^[a-zA-Z\s,.\'\-\pL]+$/u", $_POST['first_name'] ) ) {
				$data['message'] = __( 'First name can only contain letters, spaces, and hyphens.', 'uncanny-learndash-groups' );
				$data['error']   = 'invalid-first-name';
				wp_send_json_error( $data );
			}

			$first_name = (string) $_POST['first_name'];

		} elseif ( 'add-invite' === $action ) {
			$first_name      = '';
			$data['message'] = __( 'First name not received. Reload page and try again.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		// Was last name received
		if ( isset( $_POST['last_name'] ) && ( 'add-invite' === $action || 'add-leader' === $action ) ) {

			// is last name a valid name, check if it contains unicode letters only, allow hyphens
			if ( 0 === preg_match( "/^[a-zA-Z\s,.\'\-\pL]+$/u", $_POST['last_name'] ) ) {
				$data['message'] = __( 'Last name can only contain letters, spaces, and hyphens.', 'uncanny-learndash-groups' );
				$data['error']   = 'invalid-last-name';
				wp_send_json_error( $data );
			}

			$last_name = (string) $_POST['last_name'];

		} elseif ( 'add-invite' === $action ) {
			$last_name       = '';
			$data['message'] = __( 'Last name not received. Reload page and try again.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		// Was email received
		if ( isset( $_POST['email'] ) ) {

			// is email a valid name
			if ( ! is_email( $_POST['email'] ) ) {
				$data['message'] = __( 'Please enter a valid email address.', 'uncanny-learndash-groups' );
				$data['error']   = 'invalid-email';
				wp_send_json_error( $data );
			}

			$email = sanitize_email( $_POST['email'] );

		} else {
			$email           = '';
			$data['message'] = __( 'Email not received. Reload page and try again.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		if ( isset( $_POST['first_name'] ) && ! empty( $_POST['first_name'] ) ) {
			$first_name = (string) $_POST['first_name'];
		}

		if ( isset( $_POST['last_name'] ) && ! empty( $_POST['last_name'] ) ) {
			$last_name = (string) $_POST['last_name'];
		}

		$user_data = array(
			'user_login' => $email,
			'user_email' => $email,
			'first_name' => $first_name,
			'last_name'  => $last_name,
			'group_name' => $group_name
		);

		return $user_data;


	}

	/**
	 * @return array
	 */
	private function validate_new_users_data() {

		$group_name = '';

		// Was group id received
		if ( isset( $_POST['group-id'] ) ) {

			// is group a valid integer
			if ( ! absint( $_POST['group-id'] ) ) {
				$data['message'] = __( 'Group ID must be a whole number.', 'uncanny-learndash-groups' );
				$data['error']   = 'invalid-group-id';
				wp_send_json_error( $data );
			}

			$group_leader_id = get_current_user_id();
			$user_group_ids  = learndash_get_administrators_group_ids( $group_leader_id, true );

			// is the current user able to administer this group
			if ( ! in_array( $_POST['group-id'], $user_group_ids ) ) {
				$data['message'] = __( 'You do not have permission to manage this group.', 'uncanny-learndash-groups' );
				$data['error']   = 'invalid-group-id';
				wp_send_json_error( $data );
			}

			$group_name = get_the_title( absint( $_POST['group-id'] ) );

		} else {
			$data['message'] = __( 'Group ID was not received. Reload page and try again.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		// Was first name received
		if ( isset( $_POST['csv-text'] ) && '' !== $_POST['csv-text'] ) {

			// Get CSV from uploaded $_POST
			$csvArray   = self::get_csv( $_POST['csv-text'] );
			$csv_header = array_shift( $csvArray );
			$csv_header = array_map( 'trim', $csv_header );

			if ( empty( $csv_header ) ) {
				$data['message'] = __( 'The CSV file was empty or not in the correct format.', 'uncanny-learndash-groups' );
				$data['error']   = 'csv-header-empty';
				wp_send_json_error( $data );
			}

			if ( empty( $csvArray ) ) {
				$data['message'] = __( 'The CSV file was empty or not in the correct format.', 'uncanny-learndash-groups' );
				$data['error']   = 'csv-rows-empty';
				wp_send_json_error( $data );
			}

		} else {
			$data['message'] = __( 'The CSV file was empty or not in the correct format.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		// is user name header set
		if ( ! in_array( 'user_email', $csv_header ) && count( array_keys( $csv_header, 'user_email' ) ) ) {
			$data['message'] = __( 'The first row of the CSV file must contain the column header user_email.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );

		}

		// is first name header set
		if ( ! in_array( 'first_name', $csv_header ) && count( array_keys( $csv_header, 'first_name' ) ) ) {
			$data['message'] = __( 'The first row of the CSV file must contain the column header first_name.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );

		}

		// is last name header set
		if ( ! in_array( 'last_name', $csv_header ) && count( array_keys( $csv_header, 'last_name' ) ) ) {
			$data['message'] = __( 'The first row of the CSV file must contain the column header last_name.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );

		}

		$user_email_key      = array_search( 'user_email', $csv_header );
		$user_first_name_key = array_search( 'first_name', $csv_header );
		$user_last_name_key  = array_search( 'last_name', $csv_header );
		$user_pass_key       = array_search( 'user_pass', $csv_header );

		// Validate that all emails are properly formatted
		foreach ( $csvArray as $key => $row ) {
			if ( ! is_email( $row[ $user_email_key ] ) ) {
				$data['message'] = sprintf( __( 'An invalid email was detected: %s on row %s. No users were imported.', 'uncanny-learndash-groups' ), $row[ $user_email_key ], $key + 2 );
				wp_send_json_error( $data );
			}

		}

		$user_data = array();

		foreach ( $csvArray as $row ) {

			if ( false === $user_pass_key || 0 === absint( $user_pass_key ) ) {
				$user_pass = '';
			} elseif ( isset( $row[ $user_pass_key ] ) ) {
				$user_pass = $row[ $user_pass_key ];
			} else {
				$user_pass = '';
			}

			$user_data[] = array(
				'user_login' => isset( $row[ $user_email_key ] ) ? $row[ $user_email_key ] : '',
				'user_email' => isset( $row[ $user_email_key ] ) ? $row[ $user_email_key ] : '',
				'first_name' => isset( $row[ $user_first_name_key ] ) ? $row[ $user_first_name_key ] : '',
				'last_name'  => isset( $row[ $user_last_name_key ] ) ? $row[ $user_last_name_key ] : '',
				'user_pass'  => $user_pass,
				'group_name' => isset( $group_name ) ? $group_name : '',
			);
		}

		return $user_data;

	}

	/**
	 * Process CSV from file
	 *
	 * @since 1.0.0
	 */
	private function get_csv( $csv_input ) {

		$csv_input = str_replace( "\r\n", "\n", $csv_input );
		$csv_input = str_replace( "\r", "\n", $csv_input );

		$csv_input = str_getcsv( $csv_input, "\n" );
		foreach ( $csv_input as &$row ) {
			$row = str_getcsv( $row, ',' );
		}

		return $csv_input;
	}

	/**
	 * @param        $user_id
	 * @param        $plain_password
	 * @param        $welcome_template_subject
	 * @param        $welcome_template_body
	 * @param string $group_name
	 *
	 * @return bool
	 */
	function send_welcome_email( $user_id, $plain_password, $welcome_template_subject, $welcome_template_body, $group_name = '' ) {

		// Set up user data
		$user_info    = get_userdata( $user_id );
		$user_login   = $user_info->user_login;
		$user_email   = $user_info->user_email;
		$first_name   = $user_info->first_name;
		$last_name    = $user_info->last_name;
		$display_name = $user_info->display_name;

		// Filter #Username variable
		$welcome_template_subject = str_ireplace( '#Username', $user_login, $welcome_template_subject );
		$welcome_template_body    = str_ireplace( '#Username', $user_login, $welcome_template_body );

		// Filter #Email variable
		$welcome_template_subject = str_ireplace( '#Email', $user_login, $welcome_template_subject );
		$welcome_template_body    = str_ireplace( '#Email', $user_login, $welcome_template_body );

		// Filter #Password variable
		$welcome_template_subject = str_ireplace( '#Password', $plain_password, $welcome_template_subject );
		$welcome_template_body    = str_ireplace( '#Password', $plain_password, $welcome_template_body );

		// Filter #FirstName variable
		$welcome_template_subject = str_ireplace( '#FirstName', $first_name, $welcome_template_subject );
		$welcome_template_body    = str_ireplace( '#FirstName', $first_name, $welcome_template_body );

		// Filter #LastName variable
		$welcome_template_subject = str_ireplace( '#LastName', $last_name, $welcome_template_subject );
		$welcome_template_body    = str_ireplace( '#LastName', $last_name, $welcome_template_body );

		// Filter #DisplayName variable
		$welcome_template_subject = str_ireplace( '#DisplayName', $display_name, $welcome_template_subject );
		$welcome_template_body    = str_ireplace( '#DisplayName', $display_name, $welcome_template_body );

		// Filter #SiteUrl variable
		$welcome_template_subject = str_ireplace( '#SiteUrl', site_url(), $welcome_template_subject );
		$welcome_template_body    = str_ireplace( '#SiteUrl', site_url(), $welcome_template_body );

		// Filter #LoginUrl variable
		$welcome_template_subject = str_ireplace( '#LoginUrl', wp_login_url(), $welcome_template_subject );
		$welcome_template_body    = str_ireplace( '#LoginUrl', wp_login_url(), $welcome_template_body );

		// Filter #LoginUrl variable
		$welcome_template_subject = str_ireplace( '#SiteName', get_bloginfo( 'name' ), $welcome_template_subject );
		$welcome_template_body    = str_ireplace( '#SiteName', get_bloginfo( 'name' ), $welcome_template_body );

		// Filter #LoginUrl variable
		$welcome_template_subject = str_ireplace( '#GroupName', $group_name, $welcome_template_subject );
		$welcome_template_body    = str_ireplace( '#GroupName', $group_name, $welcome_template_body );

		// Remove escaped apostrophes
		$welcome_template_subject = str_replace( "\'", "'", $welcome_template_subject );
		$welcome_template_body    = str_replace( "\'", "'", $welcome_template_body );

		$user_data = [
			'user_id'    => $user_id,
			'user_email' => $user_email,
			'user_login' => $user_login,
			'first_name' => $first_name,
			'last_name'  => $last_name,
		];

		$to      = apply_filters( 'ulgm_welcome_email_to', $user_email, $user_data, $group_name );
		$subject = apply_filters( 'ulgm_welcome_email_subject', $welcome_template_subject, $user_data, $group_name );
		$body    = apply_filters( 'ulgm_welcome_email_body', $welcome_template_body, $user_data, $group_name );

		if ( ! class_exists( 'WP_Better_Emails' ) ) {
			$body = wpautop( $body );
		}

		$send_mail = apply_filters( 'ulgm_maybe_send_welcome_email', true, $to, $subject );
		if ( $send_mail ) {
			$mail_sent = wp_mail( $to, $subject, $body, $this->get_headers() );

			return $mail_sent;
		} else {
			return true;
		}
	}

	/**
	 * @param      $user_data
	 * @param      $redemption_template_subject
	 * @param      $redemption_template_body
	 * @param bool $do_not_send_emails
	 * @param bool $counter On error, return(false) or count errors(true)
	 *
	 * @return array success code || error code
	 */
	private function send_redemption_email( $user_data, $redemption_template_subject, $redemption_template_body, $do_not_send_emails = false, $counter = false ) {

		// Set up user data
		$user_email = $user_data['user_email'];
		$group_name = $user_data['group_name'];
		$first_name = $user_data['first_name'];
		$last_name  = $user_data['last_name'];

		$user = get_user_by( 'email', $user_data['user_email'] );

		$group_id = $_POST['group-id']; // already validate before this function was called

		// If the user exists and is already part of the group there is no need to send an invitation
		if ( $user && learndash_is_user_in_group( $user->ID, (int) $group_id ) ) {
			// User already exists
			if ( ! $counter ) {
				$data['message'] = __( 'This user is already a member of this group.', 'uncanny-learndash-groups' );
				wp_send_json_error( $data );
			} else {
				return array(
					'error-code' => __( 'The user on row %s is already a member of this group.', 'uncanny-learndash-groups' )
				);
			}

		}

		// Get the redemption key
		$code = SharedFunctions::get_sign_up_code_from_group_id( (int) $group_id );

		if ( ! $code ) {
			if ( ! $counter ) {
				$data['message'] = sprintf(
					__( 'User has not been added. No %s left.', 'uncanny-learndash-groups' ),
					strtolower( get_option( 'ulgm_per_seat_text_plural', 'Seats' ) )
				);
				wp_send_json_error( $data );
			} else {
				return array(
					'error-code' => sprintf(
						__( 'The user on row %s was not imported because there were no available %s left.', 'uncanny-learndash-groups' ),
						'%s', // this is parsed in another function and value should now be translated
						strtolower( get_option( 'ulgm_per_seat_text_plural', 'Seats' ) )

					)
				);
			}
		}

		// Update codes to table as with set email but user-id 0
		$added_user = SharedFunctions::set_sign_up_code_status( $code, null, null, 'not redeemed', (int) $group_id, $user_email );

		global $wpdb;

		if ( ! $added_user ) {

			if ( ! $counter ) {
				$data['message'] = __( 'The user could not be added.  Please contact the website administrator.', 'uncanny-learndash-groups' );
				wp_send_json_error( $data );
			} else {
				return array(
					'error-code' => __( 'The user on row %s could not be added.  Please contact the website administrator.', 'uncanny-learndash-groups' )
				);
			}

		}

		// Filter #email variable
		$redemption_template_subject = str_ireplace( '#FirstName', $first_name, $redemption_template_subject );
		$redemption_template_body    = str_ireplace( '#FirstName', $first_name, $redemption_template_body );

		// Filter #email variable
		$redemption_template_subject = str_ireplace( '#LastName', $last_name, $redemption_template_subject );
		$redemption_template_body    = str_ireplace( '#LastName', $last_name, $redemption_template_body );

		// Filter #email variable
		$redemption_template_subject = str_ireplace( '#Email', $user_email, $redemption_template_subject );
		$redemption_template_body    = str_ireplace( '#Email', $user_email, $redemption_template_body );

		// Filter #redemption_key variable
		$redemption_template_subject = str_ireplace( '#RedemptionKey', $code, $redemption_template_subject );
		$redemption_template_body    = str_ireplace( '#RedemptionKey', $code, $redemption_template_body );

		// Filter #SiteUrl variable
		$redemption_template_subject = str_ireplace( '#SiteUrl', site_url(), $redemption_template_subject );
		$redemption_template_body    = str_ireplace( '#SiteUrl', site_url(), $redemption_template_body );

		// Filter #LoginUrl variable
		$redemption_template_subject = str_ireplace( '#LoginUrl', wp_login_url(), $redemption_template_subject );
		$redemption_template_body    = str_ireplace( '#LoginUrl', wp_login_url(), $redemption_template_body );

		// Filter #LoginUrl variable
		$redemption_template_subject = str_ireplace( '#SiteName', get_bloginfo( 'name' ), $redemption_template_subject );
		$redemption_template_body    = str_ireplace( '#SiteName', get_bloginfo( 'name' ), $redemption_template_body );

		// Filter #LoginUrl variable
		$redemption_template_subject = str_ireplace( '#GroupName', $group_name, $redemption_template_subject );
		$redemption_template_body    = str_ireplace( '#GroupName', $group_name, $redemption_template_body );

		// Remove escaped apostrophes
		$redemption_template_subject = str_replace( "\'", "'", $redemption_template_subject );
		$redemption_template_body    = str_replace( "\'", "'", $redemption_template_body );

		$do_not_send_emails = apply_filters( 'ulgm_do_not_send_redemption_email', $do_not_send_emails, $user_data, $group_id );

		if ( $do_not_send_emails ) {

			if ( ! $counter ) {
				$data['message'] = __( 'User has been added.', 'uncanny-learndash-groups' );
				SharedFunctions::delete_transient( null, $group_id );
				wp_send_json_success( $data );
			} else {
				return array( 'success-code' => 'user-added-no-email' );
			}


		} else {
			$to      = apply_filters( 'ulgm_redemption_email_to', $user_email, $user_data, $group_id );
			$subject = apply_filters( 'ulgm_redemption_email_subject', $redemption_template_subject, $user_data, $group_id );
			$body    = apply_filters( 'ulgm_redemption_email_body', $redemption_template_body, $user_data, $group_id );

			if ( ! class_exists( 'WP_Better_Emails' ) ) {
				$body = wpautop( $body );
			}

			$send_email = apply_filters( 'ulgm_maybe_send_redemption_email', true, $to, $subject );
			do_action( 'ulgm_redemption_email_sent', $user_data, $group_id );

			if ( $send_email ) {
				$redemption_email = wp_mail( $to, $subject, $body, $this->get_headers() );

				//If the mail is successful let a a fake user and group meta
				if ( is_wp_error( $redemption_email ) ) {

					if ( ! $counter ) {
						$data['message'] = __( 'The invitation email could not be sent.  Please contact the website administrator.', 'uncanny-learndash-groups' );
						wp_send_json_success( $data );
					} else {
						return array(
							'error-code' => __( 'The invitation email could not be sent to user on row {line number}.  Please contact the website administrator.', 'uncanny-learndash-groups' )
						);
					}

				}


				if ( ! $counter ) {
					$data['message'] = sprintf( __( 'Enrollment key sent to %s.', 'uncanny-learndash-groups' ), $user_email );
					SharedFunctions::delete_transient( null, $group_id );
					wp_send_json_success( $data );
				} else {
					return array( 'success-code' => 'user-added-email-sent' );
				}
			} else {
				if ( ! $counter ) {
					$data['message'] = __( 'User has been added.', 'uncanny-learndash-groups' );
					SharedFunctions::delete_transient( null, $group_id );
					wp_send_json_success( $data );
				} else {
					return array( 'success-code' => 'user-added-no-email' );
				}
			}

		}


	}

	/**
	 * Remove users from group
	 *
	 * @since 1.0.0
	 */
	function remove_users() {

		$group_leader_id = get_current_user_id();

		// Actions permitted by the pi call (colleced from input element with name action )
		$permitted_actions = array( 'remove-users' );

		// Was an action received, and is the actions allowed
		if ( isset( $_POST['action'] ) && in_array( $_POST['action'], $permitted_actions ) ) {

			$action = (string) $_POST['action'];

		} else {
			$action          = '';
			$data['message'] = __( 'Select an action.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}


		// Does the current user have permission
		$permission = apply_filters( 'group_management_add_user_permission', 'group_leader' );
		if ( ! current_user_can( $permission ) ) {
			$data['message'] = __( 'You do not have permission to remove users.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		// Was group id received
		if ( isset( $_POST['group-id'] ) ) {

			// is group a valid integer
			if ( ! absint( $_POST['group-id'] ) ) {
				$data['message'] = __( 'Group ID must be a whole number.', 'uncanny-learndash-groups' );
				$data['error']   = 'invalid-group-id';
				wp_send_json_error( $data );
			}

			$user_group_ids = learndash_get_administrators_group_ids( $group_leader_id, true );

			// is the current user able to administer this group
			if ( ! in_array( $_POST['group-id'], $user_group_ids ) ) {
				$data['message'] = __( 'You do not have permission to manage this group.', 'uncanny-learndash-groups' );
				$data['error']   = 'invalid-group-id';
				wp_send_json_error( $data );
			}

			$group_id = absint( $_POST['group-id'] );

		} else {
			$data['message'] = __( 'Group ID was not received. Reload page and try again.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		if ( 'remove-users' === $action ) {

			if ( isset( $_POST['removing-users'] ) ) {

				$users = explode( ',', $_POST['removing-users'] );

				$temp_user_keys = array();
				$user_ids       = array();

				foreach ( $users as $user_id ) {

					if ( ! is_numeric( $user_id ) ) {
						// not a really user, we use the code as the user id if the user is not in the system... codes are always alphanumberic
						$temp_user_keys[] = $user_id;
					} else {
						// real user
						$user_ids[] = absint( $user_id );
					}

				}

				// Remove Temp users
				foreach ( $temp_user_keys as $code ) {
					SharedFunctions::remove_sign_up_code( $code, true, true );
				}

				// Remove real users from the groups
				foreach ( $user_ids as $user_id ) {


					$current_user_groups = learndash_get_users_group_ids( $user_id );

					if ( ( $key = array_search( (string) $group_id, $current_user_groups ) ) !== false ) {
						unset( $current_user_groups[ $key ] );
					}

					//$course_progress                  = get_user_meta( $user_id, '_sfwd-course_progress', true );
					$learndash_group_enrolled_courses = learndash_group_enrolled_courses( $group_id );

					// Default is not completed
					$completed = false;

					// Default progress
					$in_progress = false;

					// Check group progress courses
					foreach ( $learndash_group_enrolled_courses as $course_id ) {
						$course_progress = learndash_course_progress(
							array(
								'course_id' => $course_id,
								'user_id'   => $user_id,
								'array'     => true,
							)
						);
						// does the groups course and user progress
						if ( empty( $course_progress ) || ! is_array( $course_progress ) || ( is_array( $course_progress ) && 0 === (int) $course_progress['completed'] && 0 === (int) $course_progress['percentage'] ) ) {
							$in_progress = false;
							$completed   = false;
						} elseif ( is_array( $course_progress ) && (int) $course_progress['completed'] === (int) $course_progress['total'] ) {
							$completed = true;
						} elseif ( is_array( $course_progress ) && (int) $course_progress['total'] !== (int) $course_progress['completed'] ) {
							$in_progress = true;
							$completed   = false;
							break;
						}
					}

					// Set Status
					if ( $completed ) {
						$status = __( 'Completed', 'uncanny-learndash-groups' );
					} elseif ( $in_progress ) {
						$status = __( 'In Progress', 'uncanny-learndash-groups' );
					} else {
						$status = __( 'Not Started', 'uncanny-learndash-groups' );
					}
					//}

					if ( $status === __( 'Not Started', 'uncanny-learndash-groups' ) ) {
						$code = SharedFunctions::get_user_code( $user_id, $group_id );
						SharedFunctions::remove_sign_up_code( $code, true, true );
					}

					// set all
					learndash_set_users_group_ids( $user_id, $current_user_groups );
					do_action( 'uo-groups-role-cleanup', $user_id ); // remove the group membership role if no longer a member of any groups
					do_action( 'ulgm_group_user_removed', $user_id, (int) $group_id );
				}

				$data['message'] = 'User(s) have been removed.';
				SharedFunctions::delete_transient( null, $group_id );
				//Remove user from group
				wp_send_json_success( $data );

			} else {
				$data['message'] = __( 'Users where not received.', 'uncanny-learndash-groups' );
				wp_send_json_error( $data );
			}
		}

	}

	/**
	 * Remove users from group
	 *
	 * @param $user_id
	 * @param $group_id
	 *
	 * @since 1.0.0
	 *
	 */
	function remove_user_from_group( $user_id, $group_id ) {

		// Was group id received
		if ( ! isset( $group_id ) || ! isset( $user_id ) ) {
			return;
		}
		if ( $user_id ) {
			$existing_code = SharedFunctions::get_user_code( $user_id, $group_id );
			// Remove Temp users
			if ( ! empty( $existing_code ) ) {
				SharedFunctions::remove_sign_up_code( $existing_code, $group_id, true );
			}
			ld_update_group_access( $user_id, $group_id, true );
			do_action( 'uo-groups-role-cleanup', $user_id ); // remove the group membership role if no longer a member of any groups
		}

	}

	/**
	 * @param $user_data
	 * @param $redemption_template_subject
	 * @param $redemption_template_body
	 *
	 * @return bool
	 */
	private function resend_redemption_email( $user_data, $redemption_template_subject, $redemption_template_body ) {

		// Set up user data
		$user_email = $user_data['user_email'];
		$first_name = $user_data['first_name'];
		$last_name  = $user_data['last_name'];

		// Get the redemption key
		$redemption_key = $user_data['key'];

		// Filter #email variable
		$redemption_template_subject = str_ireplace( '#Email', $user_email, $redemption_template_subject );
		$redemption_template_body    = str_ireplace( '#Email', $user_email, $redemption_template_body );

		// Filter #first_name variable
		$redemption_template_subject = str_ireplace( '#FirstName', $first_name, $redemption_template_subject );
		$redemption_template_body    = str_ireplace( '#FirstName', $first_name, $redemption_template_body );

		// Filter #last_name variable
		$redemption_template_subject = str_ireplace( '#LastName', $last_name, $redemption_template_subject );
		$redemption_template_body    = str_ireplace( '#LastName', $last_name, $redemption_template_body );

		// Filter #redemption_key variable
		$redemption_template_subject = str_ireplace( '#RedemptionKey', $redemption_key, $redemption_template_subject );
		$redemption_template_body    = str_ireplace( '#RedemptionKey', $redemption_key, $redemption_template_body );

		// Filter #SiteUrl variable
		$redemption_template_subject = str_ireplace( '#SiteUrl', site_url(), $redemption_template_subject );
		$redemption_template_body    = str_ireplace( '#SiteUrl', site_url(), $redemption_template_body );

		// Filter #LoginUrl variable
		$redemption_template_subject = str_ireplace( '#LoginUrl', wp_login_url(), $redemption_template_subject );
		$redemption_template_body    = str_ireplace( '#LoginUrl', wp_login_url(), $redemption_template_body );

		// Remove escaped apostrophes
		$redemption_template_subject = str_replace( "\'", "'", $redemption_template_subject );
		$redemption_template_body    = str_replace( "\'", "'", $redemption_template_body );


		$to      = apply_filters( 'ulgm_redemption_email_to', $user_email, $user_data );
		$subject = apply_filters( 'ulgm_redemption_email_subject', $redemption_template_subject, $user_data );
		$body    = apply_filters( 'ulgm_redemption_email_body', $redemption_template_body, $user_data );

		if ( ! class_exists( 'WP_Better_Emails' ) ) {
			$body = wpautop( $body );
		}
		$send_mail = apply_filters( 'ulgm_maybe_resend_redemption_email', true, $to, $subject );
		if ( $send_mail ) {
			$redemption_email = wp_mail( $to, $subject, $body, $this->get_headers() );

			//If the mail is successful let a a fake user and group meta
			if ( is_wp_error( $redemption_email ) ) {
				return false;
			}
		}

		return true;


	}

	/**
	 * Adds a user automatically or sends out a redemption code
	 *
	 * @since 1.0
	 */
	function add_group_leader() {


		// Actions permitted by the pi call (colleced from input element with name action )
		$permitted_actions = array( 'add-leader' );

		// Was an action received, and is the actions allowed
		if ( isset( $_POST['action'] ) && in_array( $_POST['action'], $permitted_actions ) ) {

			$action = (string) $_POST['action'];

		} else {
			$action          = '';
			$data['message'] = __( 'Select an action.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		// Does the current user have permission
		$permission = apply_filters( 'group_management_add_group_leader_permission', 'group_leader' );
		if ( ! current_user_can( $permission ) ) {
			$data['message'] = __( 'You do not have permission to add group leaders.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		// If a single part of data doesn't validate, it dies and sends back the validation error
		$user_data = $this->validate_new_user_data( $action );
		//Utilities::log( $user_data, '$user_data', true, 'gl' );
		//Utilities::log( $action, '$action', true, 'gl' );
		$order_id = SharedFunctions::get_order_id_from_group_id( (int) $_POST['group-id'] );
		//Utilities::log( $order_id, '$order_id', true, 'gl' );

		// Add group leader and send out welcome email
		if ( 'add-leader' === $action ) {
			// Add the user and send out a welcome email NOTE group id has already been validate and the script would return error so its safe :)
			$data = $this->create_group_leader( $user_data, absint( $_POST['group-id'] ), false );
			if ( ! key_exists( 'error', $data ) ) {
				if ( 'yes' !== get_option( 'do_not_add_group_leader_as_member', 'no' ) ) {
					$user = get_user_by( 'email', $user_data['user_email'] );
					if ( 'no' === SharedFunctions::is_user_already_member_of_group( $user->ID, (int) $_POST['group-id'] ) ) {
						$user_data['user_id'] = $user->ID;

						if ( 'yes' !== get_option( 'do_not_add_group_leader_as_member', 'no' ) ) {
							$this->add_existing_user( $user_data, true, (int) $_POST['group-id'], $order_id, 'redeemed', false );
						}

						SharedFunctions::delete_transient( null, (int) $_POST['group-id'] );
					}
				}

				do_action( 'ulgm_group_leader_added', $user_data, (int) $_POST['group-id'], $order_id );

				wp_send_json_success( $data );
			} else {
				wp_send_json_error( $data );
			}
		}

	}

	/**
	 * @param      $user_data
	 * @param      $group_id
	 * @param bool $is_api
	 *
	 * @return mixed
	 */
	private function create_group_leader( $user_data, $group_id, $is_api = true ) {


		$group_leader = get_user_by( 'email', $user_data['user_email'] );

		// get grops leaders that are already in the group
		$current_group_leaders = learndash_get_groups_administrators( $group_id );

		if ( empty( $group_leader ) ) {
			// user with the email does not exist, create one
			// Create a password
			$plain_password = wp_generate_password( $length = 10, $include_standard_special_chars = false );

			// hash the password before insertion!
			$user_data['user_pass'] = $plain_password;

			// Remove new user notifications
			if ( ! function_exists( 'wp_new_user_notification' ) ) {
				function wp_new_user_notification() {
				}
			}

			// Set user as a group leader
			$user_data['role'] = 'group_leader';

			// Create the user
			$user_id = wp_insert_user( $user_data );

			// User creation falied
			if ( is_wp_error( $user_id ) ) {
				$data['message'] = __( 'Adding Group Leader failed.', 'uncanny-learndash-groups' );
				$data['error']   = json_encode( $user_id );
				if ( $is_api ) {
					wp_send_json_error( $data );
				} else {
					return $data;
				}
			}
			$is_new_user = true;
		} else {

			// User already exists, make sure its a group leader, if not the make them a group leader
			$user_id = $group_leader->ID;

			// Check if the added group leader is already part of the group
			foreach ( $current_group_leaders as $current_group_leader ) {
				if ( $current_group_leader->ID === $user_id ) {
					$data['message'] = __( 'This user is already a group leader.', 'uncanny-learndash-groups' );

					if ( $is_api ) {
						wp_send_json_success( $data );
					} else {
						return $data;
					}
				}
			}

			if ( ! user_can( $group_leader, 'group_leader' ) ) {
				$group_leader->add_role( 'group_leader' );
			}
			$is_new_user = false;
		}


		// Add new group leader to a group
		$group_leaders = array( $user_id );

		// Join new group leader and existing groups leaders
		foreach ( $current_group_leaders as $current_group_leader ) {
			$group_leaders[] = $current_group_leader->ID;
		}

		// set all groups leaders
		learndash_set_groups_administrators( $group_id, $group_leaders );

		// Name of the group
		$group_name = get_the_title( $group_id );

		if ( $is_new_user ) {
			// Group leader welcome email subject
			$ulgm_group_leader_welcome_email_subject = SharedVariables::group_leader_welcome_email_subject();

			// Group leader welcome email subject
			$ulgm_group_leader_welcome_email_body = SharedVariables::group_leader_welcome_email_body();

		} else {
			$plain_password = "";

			// Group leader welcome email subject
			$ulgm_group_leader_welcome_email_subject = SharedVariables::existing_group_leader_welcome_email_subject();

			// Group leader welcome email subject
			$ulgm_group_leader_welcome_email_body = SharedVariables::existing_group_leader_welcome_email_body();

		}

		// Send Welcome email, for extra validation we are sending in the user id and getting user data from WP because there may be filters
		$welcome_email = $this->send_welcome_email( $user_id, $plain_password, $ulgm_group_leader_welcome_email_subject, $ulgm_group_leader_welcome_email_body, $group_name );

		// Welcome Email Failed
		if ( is_wp_error( $welcome_email ) ) {
			$data['message'] = __( 'Group leader has been added. Welcome email FAILED to send.', 'uncanny-learndash-groups' );
			if ( $is_api ) {
				wp_send_json_success( $data );
			} else {
				return $data;
			}
		}

		if ( $is_new_user ) {
			$data['message'] = __( 'Group leader has been added and welcome email is sent.', 'uncanny-learndash-groups' );
			if ( $is_api ) {
				wp_send_json_success( $data );
			} else {
				return $data;
			}
		} else {
			$data['message'] = __( 'Group leader has been added and welcome email is sent. We detected that this is an existing user so first and last name have not been updated.', 'uncanny-learndash-groups' );
			if ( $is_api ) {
				wp_send_json_success( $data );
			} else {
				return $data;
			}
		}

	}

	/**
	 * Adds a user automatically or sends out a redemption code
	 *
	 * @since 1.0
	 */
	function remove_group_leaders() {

		// Actions permitted by the pi call (colleced from input element with name action )
		$permitted_actions = array( 'remove-group-leaders' );

		// Was an action received, and is the actions allowed
		if ( isset( $_POST['action'] ) && in_array( $_POST['action'], $permitted_actions ) ) {

			$action = (string) $_POST['action'];

		} else {
			$action          = '';
			$data['message'] = __( 'Select an action.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		// Does the current user have permission
		$permission = apply_filters( 'group_management_remove_group_leader_permission', 'group_leader' );
		if ( ! current_user_can( $permission ) ) {
			$data['message'] = __( 'You do not have permission to remove groups leaders.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		// Was group id received
		if ( isset( $_POST['group-id'] ) ) {

			// is group a valid integer
			if ( ! absint( $_POST['group-id'] ) ) {
				$data['message'] = __( 'Group ID must be a whole number.', 'uncanny-learndash-groups' );
				$data['error']   = 'invalid-group-id';
				wp_send_json_error( $data );
			}

			$user_group_ids = learndash_get_administrators_group_ids( get_current_user_id(), true );

			// is the current user able to administer this group
			if ( ! in_array( $_POST['group-id'], $user_group_ids ) ) {
				$data['message'] = __( 'You do not have permission to manage this group.', 'uncanny-learndash-groups' );
				$data['error']   = 'invalid-group-id';
				wp_send_json_error( $data );
			}

			$group_id = absint( $_POST['group-id'] );

			if ( 'remove-group-leaders' === $action ) {

				if ( isset( $_POST['removing-group-leaders'] ) ) {
					$group_leaders_new = array();
					$users             = explode( ',', $_POST['removing-group-leaders'] );

					$current_group_leaders     = learndash_get_groups_administrators( $group_id, true );
					$current_group_leaders_ids = wp_list_pluck( $current_group_leaders, 'ID' );

					$users                    = array_map( 'intval', $users );
					$group_leaders_new_object = array_diff( $current_group_leaders_ids, $users );

					// convert to pure array, its what LD expects
					foreach ( $group_leaders_new_object as $user_id ) {
						$group_leaders_new[] = $user_id;
					}

					// never delete the admin group leader
//					$admin_group_leader = get_post_meta( $group_id, 'locked_admin_group_leader', true );

//					if ( 0 !== absint( $admin_group_leader ) ) {
//						$group_leaders_new[] = absint( $admin_group_leader );
//					}

					if ( ! in_array( get_current_user_id(), $group_leaders_new ) ) {
						$group_leaders_new[] = absint( get_current_user_id() );
					}

//					if ( 1 === count( $group_leaders_new ) && (int) $admin_group_leader === (int) $group_leaders_new[0] ) {
//						$data['message'] = __( 'This Group Leader cannot be deleted.', 'uncanny-learndash-groups' );
//						wp_send_json_error( $data );
//					}

					learndash_set_groups_administrators( $group_id, $group_leaders_new );
					if ( has_action( 'uo-groups-role-cleanup' ) ) {
						$group_leaders_remove = array_diff( $current_group_leaders_ids, array_intersect( $group_leaders_new, $current_group_leaders_ids ) );
						if ( ! empty( $group_leaders_remove ) ) {
							foreach ( $group_leaders_remove as $user_id ) {
								do_action( 'ulgm_group_leader_removed', $user_id, (int) $group_id );
								do_action( 'uo-groups-role-cleanup', $user_id ); // remove the group leader role if no longer a member of any groups
							}
						}
					}
					$data['message'] = __( 'User(s) have been removed.', 'uncanny-learndash-groups' );
					if ( ! empty( $users ) ) {
						foreach ( $users as $user_id ) {
							SharedFunctions::delete_transient( $user_id );
						}
					}
					SharedFunctions::delete_transient( null, $group_id );
					//Remove user from group
					wp_send_json_success( $data );

				} else {
					$data['message'] = __( 'Users where not received.', 'uncanny-learndash-groups' );
					wp_send_json_error( $data );
				}
			}

		} else {
			$data['message'] = __( 'Group ID was not received. Reload page and try again.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}


	}

	/**
	 * Save email templates for user redemption email, user welcome email and group leader welcome email
	 * @since 1.0.0
	 *
	 */
	function save_general_settings() {

		// Actions permitted by the pi call (colleced from input element with name action )
		$permitted_actions = array( 'save-general-settings' );

		// Was an action received, and is the actions allowed
		if ( isset( $_POST['action'] ) && in_array( $_POST['action'], $permitted_actions ) ) {

			$action = (string) $_POST['action'];

		} else {
			$action          = '';
			$data['message'] = __( 'Select an action.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		// Does the current user have permission
		$permission = apply_filters( 'general_settings_update_permission', 'manage_options' );

		if ( ! current_user_can( $permission ) ) {
			$data['message'] = __( 'You do not have permission to save settings.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		if ( isset( $_POST['ulgm_term_condition'] ) ) {
			update_option( 'ulgm_term_condition', $_POST['ulgm_term_condition'] );
		}

		if ( isset( $_POST['ulgm_group_management_page'] ) ) {
			update_option( 'ulgm_group_management_page', $_POST['ulgm_group_management_page'] );
		}

		if ( isset( $_POST['ulgm_group_buy_courses_page'] ) ) {
			update_option( 'ulgm_group_buy_courses_page', $_POST['ulgm_group_buy_courses_page'] );
		}

		if ( isset( $_POST['ulgm_group_report_page'] ) ) {
			update_option( 'ulgm_group_report_page', $_POST['ulgm_group_report_page'] );
		}

		if ( isset( $_POST['ulgm_group_assignment_report_page'] ) ) {
			update_option( 'ulgm_group_assignment_report_page', $_POST['ulgm_group_assignment_report_page'] );
		}

		if ( isset( $_POST['ulgm_group_quiz_report_page'] ) ) {
			update_option( 'ulgm_group_quiz_report_page', $_POST['ulgm_group_quiz_report_page'] );
		}

		if ( isset( $_POST['ulgm_group_manage_progress_page'] ) ) {
			update_option( 'ulgm_group_manage_progress_page', $_POST['ulgm_group_manage_progress_page'] );
		}

		if ( isset( $_POST['ulgm_group_essay_report_page'] ) ) {
			update_option( 'ulgm_group_essay_report_page', $_POST['ulgm_group_essay_report_page'] );
		}

		if ( isset( $_POST['ulgm_group_license_product_cat'] ) ) {
			update_option( 'ulgm_group_license_product_cat', $_POST['ulgm_group_license_product_cat'] );
		}

		if ( isset( $_POST['ulgm_main_color'] ) ) {
			update_option( 'ulgm_main_color', $_POST['ulgm_main_color'] );
		}

		if ( isset( $_POST['ulgm_font_color'] ) ) {
			update_option( 'ulgm_font_color', $_POST['ulgm_font_color'] );
		}

		if ( isset( $_POST['ulgm_add_to_cart_message'] ) ) {
			update_option( 'ulgm_add_to_cart_message', $_POST['ulgm_add_to_cart_message'] );
		}

		if ( isset( $_POST['ulgm_per_seat_text'] ) ) {
			update_option( 'ulgm_per_seat_text', $_POST['ulgm_per_seat_text'] );
		}

		if ( isset( $_POST['ulgm_per_seat_text_plural'] ) ) {
			update_option( 'ulgm_per_seat_text_plural', $_POST['ulgm_per_seat_text_plural'] );
		}

		if ( isset( $_POST['show_license_product_on_front'] ) ) {
			update_option( 'show_license_product_on_front', $_POST['show_license_product_on_front'] );
		} else {
			delete_option( 'show_license_product_on_front' );
		}

		if ( isset( $_POST['allow_to_remove_users_anytime'] ) ) {
			update_option( 'allow_to_remove_users_anytime', $_POST['allow_to_remove_users_anytime'] );
		} else {
			delete_option( 'allow_to_remove_users_anytime' );
		}

		if ( isset( $_POST['allow_group_leaders_to_manage_progress'] ) ) {
			update_option( 'allow_group_leaders_to_manage_progress', $_POST['allow_group_leaders_to_manage_progress'] );
		} else {
			delete_option( 'allow_group_leaders_to_manage_progress' );
		}

		if ( isset( $_POST['group_leaders_dont_use_seats'] ) ) {
			update_option( 'group_leaders_dont_use_seats', $_POST['group_leaders_dont_use_seats'] );
		} else {
			delete_option( 'group_leaders_dont_use_seats' );
		}

		if ( isset( $_POST['do_not_add_group_leader_as_member'] ) ) {
			update_option( 'do_not_add_group_leader_as_member', $_POST['do_not_add_group_leader_as_member'] );
		} else {
			delete_option( 'do_not_add_group_leader_as_member' );
		}

		if ( isset( $_POST['add_courses_as_part_of_license'] ) ) {
			update_option( 'add_courses_as_part_of_license', $_POST['add_courses_as_part_of_license'] );
		} else {
			delete_option( 'add_courses_as_part_of_license' );
		}

		if ( isset( $_POST['add_groups_as_woo_products'] ) ) {
			update_option( 'add_groups_as_woo_products', $_POST['add_groups_as_woo_products'] );
		} else {
			delete_option( 'add_groups_as_woo_products' );
		}

		$data['message'] = __( 'Settings have been saved.', 'uncanny-learndash-groups' );
		wp_send_json_success( $data );

	}

	/**
	 * Save email templates for user redemption email, user welcome email and group leader welcome email
	 * @since 1.0.0
	 *
	 */
	function save_email_templates() {

		// Actions permitted by the pi call (colleced from input element with name action )
		$permitted_actions = array( 'save-email-templates' );

		// Was an action received, and is the actions allowed
		if ( isset( $_POST['action'] ) && in_array( $_POST['action'], $permitted_actions ) ) {

			$action = (string) $_POST['action'];

		} else {
			$action          = '';
			$data['message'] = __( 'Select an action.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		// Does the current user have permission
		$permission = apply_filters( 'email_template_update_permission', 'manage_options' );

		if ( ! current_user_can( $permission ) ) {
			$data['message'] = __( 'You do not have permission to save settings.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		if ( isset( $_POST['ulgm_invitation_user_email_subject'] ) ) {
			update_option( 'ulgm_invitation_user_email_subject', $_POST['ulgm_invitation_user_email_subject'] );
		}

		if ( isset( $_POST['ulgm_invitation_user_email_body'] ) ) {
			update_option( 'ulgm_invitation_user_email_body', $_POST['ulgm_invitation_user_email_body'] );
		}

		if ( isset( $_POST['ulgm_user_welcome_email_subject'] ) ) {
			update_option( 'ulgm_user_welcome_email_subject', $_POST['ulgm_user_welcome_email_subject'] );
		}

		if ( isset( $_POST['ulgm_user_welcome_email_body'] ) ) {
			update_option( 'ulgm_user_welcome_email_body', $_POST['ulgm_user_welcome_email_body'] );
		}

		if ( isset( $_POST['ulgm_existing_user_welcome_email_subject'] ) ) {
			update_option( 'ulgm_existing_user_welcome_email_subject', $_POST['ulgm_existing_user_welcome_email_subject'] );
		}

		if ( isset( $_POST['ulgm_existing_user_welcome_email_body'] ) ) {
			update_option( 'ulgm_existing_user_welcome_email_body', $_POST['ulgm_existing_user_welcome_email_body'] );
		}

		if ( isset( $_POST['ulgm_group_leader_welcome_email_subject'] ) ) {
			update_option( 'ulgm_group_leader_welcome_email_subject', $_POST['ulgm_group_leader_welcome_email_subject'] );
		}

		if ( isset( $_POST['ulgm_group_leader_welcome_email_body'] ) ) {
			update_option( 'ulgm_group_leader_welcome_email_body', $_POST['ulgm_group_leader_welcome_email_body'] );
		}

		if ( isset( $_POST['ulgm_existing_group_leader_welcome_email_subject'] ) ) {
			update_option( 'ulgm_existing_group_leader_welcome_email_subject', $_POST['ulgm_existing_group_leader_welcome_email_subject'] );
		}

		if ( isset( $_POST['ulgm_existing_group_leader_welcome_email_body'] ) ) {
			update_option( 'ulgm_existing_group_leader_welcome_email_body', $_POST['ulgm_existing_group_leader_welcome_email_body'] );
		}

		if ( isset( $_POST['ulgm_email_from'] ) ) {
			update_option( 'ulgm_email_from', $_POST['ulgm_email_from'] );
		}

		if ( isset( $_POST['ulgm_name_from'] ) ) {
			update_option( 'ulgm_name_from', $_POST['ulgm_name_from'] );
		}

		if ( isset( $_POST['ulgm_reply_to'] ) ) {
			update_option( 'ulgm_reply_to', $_POST['ulgm_reply_to'] );
		}

		$data['message'] = __( 'Email settings have been saved.', 'uncanny-learndash-groups' );
		wp_send_json_success( $data );

	}

	/**
	 * Save email templates for user redemption email, user welcome email and group leader welcome email
	 * @since 1.0.0
	 *
	 */
	function download_keys_csv() {

		// Actions permitted by the pi call (colleced from input element with name action )
		$permitted_actions = array( 'download' );

		// Was an action received, and is the actions allowed
		if ( isset( $_POST['action'] ) && in_array( $_POST['action'], $permitted_actions ) ) {

			$action = (string) $_POST['action'];

		} else {
			$action          = '';
			$data['message'] = __( 'Select an action.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		// Does the current user have permission
		$permission = apply_filters( 'download_keys_csv_permission', 'group_leader' );
		if ( ! current_user_can( $permission ) ) {
			$data['message'] = __( 'You do not have permission to download csv keys.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		$group_leader_id = get_current_user_id();
		$user_group_ids  = learndash_get_administrators_group_ids( $group_leader_id, true );

		// is the current user able to administer this group
		if ( ! in_array( $_POST['group-id'], $user_group_ids ) ) {
			$data['message'] = __( 'You do not have permission to manage this group.', 'uncanny-learndash-groups' );
			$data['error']   = 'invalid-group-id';
			wp_send_json_error( $data );
		}

		$group_id = absint( $_POST['group-id'] );

		$users = apply_filters( 'ulgm_download_users_keys', $this->get_unused__key_users_data( $group_id ), $group_id );
		//Utilities::log( $users, '$users', true, 'users' );
		$csv = '';

		if ( ! empty( $users ) ) {
			$headers = apply_filters( 'ulgm_download_keys_header', "Group,Key\n", $group_id );
			$csv     .= $headers;
			foreach ( $users as $row ) {
				$csv .= implode( ',', $row ) . "\n";
			}
		} else {
			$headers = apply_filters( 'ulgm_download_keys_header', "Group,Key\n", $group_id );
			$csv     .= $headers;
		}

		// File name
		$group_slug = get_post_field( 'post_name', $group_id );
		$file_name  = 'keys-' . $group_slug . '-' . date( 'Y-m-d' );
		$file_name  = apply_filters( 'csv_file_name', $file_name, $group_slug, $group_id, $group_leader_id );

		/// Trigger file creation to frontend
		$data['reload']        = false;
		$data['call_function'] = 'downloadCsv';
		$data['function_vars'] = array(
			'csvDataString' => $csv,
			'fileName'      => $file_name

		);

		wp_send_json_success( $data );
	}

	/**
	 * Get and globalize user data
	 *
	 * @param $group_id
	 *
	 * @return array
	 * @since 1.0.0
	 *
	 */
	private function get_unused__key_users_data( $group_id ) {

		$unused__key_users_data = array();
		$get_codes              = SharedFunctions::get_codes_for_download_csv( $group_id );
		//Utilities::log( $get_codes, '$get_codes', true, 'get_codes' );
		if ( $get_codes ) {
			foreach ( $get_codes as $code ) {
				/*if ( ! empty( $code['student_id'] ) && absint( $code['student_id'] ) ) {
					$user       = get_user_by( 'ID', $code['student_id'] );
					$user_email = $user->user_email;
				} elseif ( ! empty( $code['user_email'] ) && is_email( $code['user_email'] ) ) {
					$user_email = $code['user_email'];
				} else {
					$user_email = '';
				}*/

				$unused__key_users_data[] = array(
					$code['group_name'],
					$code['code'],
					//$code['code_status'],
					//$code['used_date'],
					//$user_email,
				);
			}
		}

		return $unused__key_users_data;
	}

	/**
	 * @param bool $group_users_email
	 * @param array $group_leader_details
	 *
	 * @return array|string
	 */
	private function get_headers( $group_users_email = false, $group_leader_details = [] ) {

		$headers = [];
		if ( ! $group_users_email ) {
			$from_email = get_option( 'ulgm_email_from', '' );
			$from_name  = get_option( 'ulgm_name_from', '' );
			$reply_to   = get_option( 'ulgm_reply_to', '' );

			$default_admin_email = get_option( 'admin_email', '' );
			$default_blogname    = get_option( 'blogname', '' );

			if ( empty( $from_name ) ) {

				if ( empty( $from_email ) ) {
					$headers[] = "From: $default_blogname <$default_admin_email>";
				} else {
					$headers[] = "From: $default_blogname <$from_email>";
				}

				if ( empty( $reply_to ) ) {
					// don't set any headers
				} else {
					$headers[] = "Reply-To: $default_blogname <$reply_to>";
				}

			} else {

				if ( empty( $from_email ) ) {
					$headers[] = "From: $from_name <$default_admin_email>";
				} else {
					$headers[] = "From: $from_name <$from_email>";
				}

				if ( empty( $reply_to ) ) {
					// don't set any headers
				} else {
					$headers[] = "Reply-To: $from_name <$reply_to>";
				}

			}
		} else {

			$from_name  = $group_leader_details['from_name'];
			$from_email = $group_leader_details['from_email'];
			$reply_to   = $group_leader_details['reply_to'];

			$headers[] = "From: $from_name <$from_email>";
			$headers[] = "Reply-To: $from_name <$reply_to>";

		}

		if ( ! class_exists( 'WP_Better_Emails' ) ) {
			$headers[] = 'Content-Type: text/html; charset=UTF-8';
		}


		if ( empty( $headers ) ) {
			return '';
		}

		return $headers;

	}
}