<?php
/**
 * Plugin Name:     LearnDash EDD Integration Plugin
 * Plugin URI:      http://learndash.com
 * Description:     LearnDash integration plugin for EDD
 * Version:         1.2.0
 * Author:          LearnDash
 * Author URI:      http://learndash.com
 * Text Domain:     learndash-edd
 * Doman Path: 		/languages/
 *
 * @package         LearnDash\EDD
 */

// Exit if accessed directly
if( ! defined( 'ABSPATH' ) ) exit;


if( ! class_exists( 'LearnDash_EDD' ) ) {


    /**
     * Main LearnDash_EDD class
     *
     * @since       1.0.0
     */
    class LearnDash_EDD {


        /**
         * @var         LearnDash_EDD $instance The one true LearnDash_EDD
         * @since       1.0.0
         */
        private static $instance;


        /**
         * Get active instance
         *
         * @access      public
         * @since       1.0.0
         * @return      self::$instance The one true LearnDash_EDD
         */
        public static function instance() {
            if( ! self::$instance ) {
                self::$instance = new LearnDash_EDD();
                self::$instance->setup_constants();
                self::$instance->includes();

                add_action( 'plugins_loaded', array( 'LearnDash_EDD', 'load_textdomain' ) );
                
                self::$instance->hooks();
            }

            return self::$instance;
        }


        /**
         * Setup plugin constants
         *
         * @access      private
         * @since       1.0.0
         * @return      void
         */
        private function setup_constants() {
            // Plugin version
            define( 'LEARNDASH_EDD_VERSION', '1.1.1' );

            // Plugin path
            define( 'LEARNDASH_EDD_DIR', plugin_dir_path( __FILE__ ) );

            // Plugin URL
            define( 'LEARNDASH_EDD_URL', plugin_dir_url( __FILE__ ) );
        }


        /**
         * Include required files
         *
         * @access      private
         * @since       1.0.0
         * @return      void
         */
        private function includes() {
            require_once LEARNDASH_EDD_DIR . 'includes/scripts.php';

            if( is_admin() ) {
                require_once LEARNDASH_EDD_DIR . 'includes/metabox.php';
            }
			
			// Prior to this release we have some debug output being written to a log file in the site root. 
			// This code is added to remove the file. 
			if ( file_exists( ABSPATH.'ld-edd.log' ) ) {
				@unlink( ABSPATH.'ld-edd.log' );
			}
        }


        /**
         * Run action and filter hooks
         *
         * @access      private
         * @since       1.0.0
         * @return      void
         */
        private function hooks() {
            // Add meta boxes

            // Update courses
            add_action( 'edd_updated_edited_purchase', array( $this, 'updated_edited_purchase' ) );
            add_action( 'edd_complete_purchase', array( $this, 'complete_purchase' ) );
            add_action( 'edd_recurring_update_subscription', array( $this, 'update_subscription' ), 10, 3 );

            // Remove courses
            add_action( 'edd_subscription_cancelled', array( $this, 'cancel_subscription' ), 10, 2 );
            add_action( 'edd_subscription_expired', array( $this, 'cancel_subscription' ), 10, 2 );
            add_action( 'edd_update_payment_status', array( $this, 'remove_access_on_payment_update' ), 10, 3 );
            add_action( 'edd_payment_delete', array( $this, 'remove_access_on_payment_delete' ), 10, 1 );
        }


        /**
         * Internationalization
         *
         * @access      public
         * @since       1.0.0
         * @return      void
         */
        public function load_textdomain() {
            // Set filter for language directory
            $lang_dir = dirname( plugin_basename( __FILE__ ) ) . '/languages/';
            $lang_dir = apply_filters( 'learndash_edd_language_directory', $lang_dir );

            // Traditional WordPress plugin locale filter
            $locale = apply_filters( 'plugin_locale', get_locale(), '' );
            $mofile = sprintf( '%1$s-%2$s.mo', 'learndash-edd', $locale );

            // Setup paths to current locale file
            $mofile_local   = $lang_dir . $mofile;
            $mofile_global  = WP_LANG_DIR . '/learndash-edd' . $mofile;

            if( file_exists( $mofile_global ) ) {
                // Look in global /wp-content/languages/learndash-edd/ folder
                load_textdomain( 'learndash-edd', $mofile_global );
            } elseif( file_exists( $mofile_local ) ) {
                // Look in local /wp-content/plugins/learndash-edd/languages/ folder
                load_textdomain( 'learndash-edd', $mofile_local );
            } else {
                // Load the default language files
                load_plugin_textdomain( 'learndash-edd', false, $lang_dir );
            }

            // include translation/update class
            include LEARNDASH_EDD_DIR . 'includes/class-translations-ld-edd.php';
        }


        /**
         * Update LearnDash course access
         *
         * @access      public
         * @since       1.0.0
         * @return      void
         */
        public function update_course_access( $transaction_id, $remove = false ) {

            // Get transaction data
			$transaction = get_post( $transaction_id );
			if ($transaction) {
				
				$edd_payment_meta = get_post_meta( $transaction_id, '_edd_payment_meta', true );
				if ( ( isset( $edd_payment_meta['downloads'] ) ) 
				 && ( is_array( $edd_payment_meta['downloads'] ) ) 
				 && ( !empty( $edd_payment_meta['downloads'] ) ) ) {
					
					foreach( $edd_payment_meta['downloads'] as $download ) {
						
						if ( isset( $download['id'] ) ) {
						
							$download_id = intval($download['id']);
							if (!empty( $download_id ) ) {
						
								// Get customer ID.
								$customer_id 	= get_post_meta( $transaction_id, '_edd_payment_user_id', true );
								
								if (!empty($customer_id)) {

									// Get the Courses 
									$courses = get_post_meta( $download_id, '_edd_learndash_course', true );

									if ( ( is_array( $courses ) ) && ( ! empty( $courses ) ) ) {
										foreach( $courses as $course_id ) {
											ld_update_course_access( (int) $customer_id, (int) $course_id, $remove );
										}
									}
								}
							}
						}
					}
				}
			}
        }

        /**
         * Give course access when user complete a purchase
         * 
         * @param  int  $payment_id   ID of a transaction
         * @since  1.1.0
         */
        public function complete_purchase( $payment_id ) {
            $this->update_course_access( $payment_id );
        }

        /**
         * Give course access when payment is updated to complete on payment edit screen
         * 
         * @param  int  $payment_id   ID of a payment
         * @since  1.1.0
         */
        public function updated_edited_purchase( $payment_id ) {
            $payment = new EDD_Payment( $payment_id );

            if ( $payment->status != 'publish' && $payment->status != 'edd_subscription' ) {
                return;
            }

            $this->update_course_access( $payment_id );
        }

        /**
         * Remove course access when payment is refunded
         * 
         * @param  object $payment EDD_Payment object
         * @since  1.1.0
         */
        public function refund_payment( $payment ) {
            $transaction_id = $payment->transaction_id;

            $this->update_course_access( $transaction_id, $remove = true );
        }

        /**
         * Update course access when subscription is completed
         *
         * @param  int    $subscription_id  ID of a subscription
         * @param  object $subscription     EDD_Subscription object
         * @since  1.1.0
         */
        public function update_subscription( $subscription_id, $args, $subscription )
        {
            if ( $args['status'] != 'active' && $args['status'] != 'completed' ) {
                return;
            }

            $transaction_id = $subscription->parent_payment_id;

            $this->update_course_access( $transaction_id );
        }

        /**
         * Remove course access when subscription is cancelled
         *
         * @param  int    $subscription_id  ID of a subscription
         * @param  object $subscription     EDD_Subscription object
         * @since  1.1.0
         */
        public function cancel_subscription( $subscription_id, $subscription ) {
            $transaction_id = $subscription->parent_payment_id;

            $this->update_course_access( $transaction_id, $remove = true );
        }

        /**
         * Remove user course access when EDD payment status is updated to 
         * other than completed and renewal
         * 
         * @param  int    $payment_id ID of the payment
         * @param  string $status     New status
         * @param  string $old_status Old status
         */
        public function remove_access_on_payment_update( $payment_id, $status, $old_status ) {
            if ( $status == 'complete' || $status == 'edd_subscription' ) {
                return;
            }

            $this->update_course_access( $payment_id, $remove = true );
        }

        /**
         * Remove user course access when EDD payment status is deleted
         * 
         * @param  int    $payment_id ID of the payment
         */
        public function remove_access_on_payment_delete( $payment_id ) {
            $this->update_course_access( $payment_id, $remove = true );
        }
    }
}


/**
 * The main function responsible for returning the one true LearnDash_EDD
 * instance to functions everywhere
 *
 * @since       1.0.0
 * @return      LearnDash_EDD The one true LearnDash_EDD
 */
function learndash_edd_load() {
    return LearnDash_EDD::instance();
}
add_action( 'plugins_loaded', 'learndash_edd_load' );
