<?php
/**
 * Handles plugin upgrades.
 *
 * @since 1.1.0
 */

defined( 'ABSPATH' ) || die();

/**
 * Class LD_ConvertKit_Upgrade
 *
 * Handles plugin upgrades.
 *
 * @since 1.1.0
 */
class LD_ConvertKit_Upgrade {

	/**
	 * LD_ConvertKit_Upgrade constructor.
	 *
	 * @since 1.1.0
	 *
	 * @return bool True if needs to upgrade, false if does not.
	 */
	function __construct() {

		add_action( 'admin_init', array( $this, 'check_upgrades' ) );
	}

	/**
	 * Checks for upgrades and migrations.
	 *
	 * @since 1.1.0
	 * @access private
	 */
	function check_upgrades() {

		$version = get_option( 'ld_convertkit_version', 0 );

		if ( version_compare( $version, LEARNDASH_CONVERTKIT_VER ) === - 1 ) {

			update_option( 'ld_convertkit_version', LEARNDASH_CONVERTKIT_VER );
		}

		$last_upgrade = get_option( 'ld_convertkit_last_upgrade', 0 );

		$upgrades = $this->get_upgrades();

		end( $upgrades );
		$final_upgrade = key( $upgrades );

		if ( version_compare( $last_upgrade, $final_upgrade ) === - 1 ) {

			$this->do_upgrades();
		}
	}

	/**
	 * Runs upgrades.
	 *
	 * @since 1.1.0
	 * @access private
	 */
	function do_upgrades() {

		$last_upgrade = get_option( 'ld_convertkit_last_upgrade', 0 );

		foreach ( $this->get_upgrades() as $upgrade_version => $upgrade_callback ) {

			if ( version_compare( $last_upgrade, $upgrade_version ) === - 1 ) {

				$result = call_user_func( $upgrade_callback );

				if ( $result === true ) {

					update_option( 'ld_convertkit_last_upgrade', $upgrade_version );
				}
			}
		}
	}

	/**
	 * Returns an array of all versions that require an upgrade.
	 *
	 * @since 1.1.0
	 * @access private
	 *
	 * @return array
	 */
	function get_upgrades() {

		return array(
			'1.1.0' => array( $this, 'upgrade_1_1_0' ),
		);
	}

	/**
	 * 1.1.0 upgrade script.
	 *
	 * - Adds the ConvertKit subscriber ID to the users meta for all active subscribers.
	 *
	 * @since 1.1.0
	 * @access private
	 */
	function upgrade_1_1_0() {

		if ( ! LD_ConvertKit_API() ) {

			return false;
		}

		$subscriber_form = ld_convertkit_get_option_field( 'subscriber_form' );

		if ( ! $subscriber_form ) {

			return true;
		}

		ld_convertkit_sync_subscribers( false );

		return true;
	}
}